local S = imese.translator
local id = imese.id
local get_compatibility_item = imese.get_compatibility_item


-- Get compatibility tables
local c_mold = get_compatibility_item('mese_crystal_mold')
local c_mold_broken = get_compatibility_item('mese_crystal_mold_broken')
local c_powder = get_compatibility_item('mese_powder')
local c_agent = get_compatibility_item('cutting_agent')


-- Mese Crystal Mold
--
-- The mold is needed to shape the crystal in later steps. As material for the
-- mold four default steel ingot are used. They are placed in a rhombus shape
-- in the crafting inventory.
--
--  ┌───┬───┬───┐
--  │ o │ x │   │
--  ├───┼───┼───┤
--  │ x │   │ x │
--  ├───┼───┼───┤
--  │   │ x │   │
--  └───┴───┴───┘
--
--  The `x` marks the position of *Minetest Game*’s default mod steel ingots.
--  The optional compatibility item `o` is used when set.

local mold_texture = ('(+base^[combine:16x16:0,0=+mask)'):gsub('%+%w+', {
    ['+base'] = '(default_steel_block.png^[transformR90)',
    ['+mask'] = 'imese_mese_mold_mask.png',
})..'^[makealpha:0,0,0'

core.register_craftitem(id.crystal_mold, {
    description = S('Mese Crystal Mold'),
    inventory_image = mold_texture
})

core.register_craft({
    output = id.crystal_mold..' 4',
    recipe = {
        { c_mold.item,           'default:steel_ingot', ''                    },
        { 'default:steel_ingot', '',                    'default:steel_ingot' },
        { '',                    'default:steel_ingot', ''                    }
    },
    replacements = c_mold.replacements
})

core.register_craft({
    output = 'default:steel_ingot',
    type = 'shapeless',
    recipe = { id.crystal_mold }
})


-- Broken Mese Crystal Mold
--
-- The broken mold is returned on crafting a wet industrial Mese crystal. The
-- broken mold of course cannot be crafted.

local broken_mold_texture = ('(+base^[combine:16x16:0,0=+mask)'):gsub('%+%w+', {
    ['+base'] = '(default_steel_block.png^[transformR90)',
    ['+mask'] = 'imese_mese_mold_broken_mask.png',
})..'^[makealpha:0,0,0'

core.register_craftitem(id.crystal_mold_broken, {
    description = S('Broken Mese Crystal Mold'),
    inventory_image = broken_mold_texture
})

core.register_craft({
    output = 'default:steel_ingot',
    recipe = {
        { id.crystal_mold_broken, id.crystal_mold_broken, '' },
        { id.crystal_mold_broken, id.crystal_mold_broken, '' },
        { '',                     '',                     '' }
    }
})


-- Mese Powder
--
-- This is the base ingredient for the material that will be used to create
-- the crystal. It can be created by putting a Mese crystal fragment in the
-- crafting grid.
--
-- If the compatibility item is set then this items needs to be used here, too.

local powder_texture = ('(+base^[combine:16x16:0,0=+mask)'):gsub('%+%w+', {
    ['+base'] = '(default_mese_block.png^[transform:FXR90)',
    ['+mask'] = 'imese_mese_powder_mask.png',
})..'^[makealpha:0,0,0'


core.register_craftitem(id.crystal_powder, {
    description = S('Mese Crystal Powder'),
    inventory_image = powder_texture
})

core.register_craft({
    output = id.crystal_powder..' 1',
    type = 'shapeless',
    recipe = {
        'default:mese_crystal_fragment',
        (c_powder.item ~= '' and c_powder.item or nil)
    },
    replacements = c_powder.replacements
})


-- Cutting Agent
--
-- This item is used to dilute the created Mese powder. The ID is not specific
-- to another item but by code cobblestone is used. It needs to be laid out in
-- the crafting grid as shown below.
--
--  ┌───┬───┬───┐
--  │ x │ o │ x │
--  ├───┼───┼───┤
--  │   │   │   │
--  ├───┼───┼───┤
--  │ x │   │   │
--  └───┴───┴───┘
--
--  The `x` marks the position of the cutting agent material. The optional
--  compatibility item `o` is used when set.

local agent_texture = ('(+base^[combine:16x16:0,0=+mask)'):gsub('%+%w+', {
    ['+base'] = 'default_cobble.png',
    ['+mask'] = 'imese_separated_cobblestone_mask.png',
})..'^[makealpha:0,0,0'

core.register_craftitem(id.cutting_agent, {
    description = S('Separated Cobblestone'),
    inventory_image = agent_texture
})

core.register_craft({
    output = id.cutting_agent..' 9',
    recipe = {
        { 'default:cobble', c_agent.item, 'default:cobble' },
        { '',               '', '' },
        { 'default:cobble', '', '' }
    },
    replacements = c_agent.replacements
})

core.register_craft({
    output = 'default:cobble',
    type = 'shapeless',
    recipe = {
        id.cutting_agent,
        id.cutting_agent,
        id.cutting_agent
    }
})


-- Create Raw Material
--
-- The material to create the crystal is created in two steps.
--
--   1. In a shapeless recipe combine the created crystal powder and the
--      created cutting agent
--
--   2. In a shapeless recipe combine the item from step 1 with a bucket of
--      water (the empty bucket is returned)

local water_drops = ('(+water^+mask)^[makealpha:0,0,0'):gsub('%+%w+', {
    ['+water'] = 'default_water.png',
    ['+mask'] = '(imese_mese_powder_mask.png^[transformR90)',
})

core.register_craftitem(id.diluted_powder_dry, {
    description = S('Dry Diluted Mese Powder'),
    inventory_image = ('(+agent)^(+powder)'):gsub('%+%w+', {
        ['+agent'] = agent_texture,
        ['+powder'] = powder_texture
    })
})

core.register_craftitem(id.diluted_powder_wet, {
    description = S('Wet Diluted Mese Powder'),
    inventory_image = ('(+water)^(+agent)^(+powder)'):gsub('%+%w+', {
        ['+agent'] = agent_texture,
        ['+powder'] = powder_texture,
        ['+water'] = water_drops
    })
})

core.register_craft({
    output = id.diluted_powder_dry,
    type = 'shapeless',
    recipe = { id.crystal_powder, id.cutting_agent }
})

core.register_craft({
    output = id.diluted_powder_wet,
    type = 'shapeless',
    recipe = { id.diluted_powder_dry, 'bucket:bucket_water' },
    replacements = { { 'bucket:bucket_water', 'bucket:bucket_empty' } }
})

-- TODO: Needs to be changed -> Use 3rd-party registry file?
if core.get_modpath("bucket_wooden") then
    core.register_craft({
        output = id.diluted_powder_wet,
        type = 'shapeless',
        recipe = { id.diluted_powder_dry, 'group:water_bucket_wooden' },
        replacements = { { 'group:water_bucket_wooden', 'bucket_wooden:bucket_empty' } }
    })
end


-- Molding
--
-- To mold a crystal put together the Mese crystal mold and the wet diluted
-- Mese powder in a shapeless recipe. A molded crystal is returned and a broken
-- mold is returned.
--
-- 4 Broken Molds in a 2x2 shape can be crafted into one steel ingot.

core.register_craftitem(id.molded_crystal, {
    description = S('Wet Molded Mese Crystal'),
    inventory_image = ('(+base)^[makealpha:0,0,0'):gsub('%+%w+', {
        ['+base'] = 'default_mese_crystal.png^[colorize:#00000055',
    })
})

core.register_craft({
    output = id.molded_crystal,
    type = 'shapeless',
    recipe = { id.diluted_powder_wet, id.crystal_mold },
    replacements = { {id.crystal_mold,id.crystal_mold_broken..' 1'} }
})


-- Burning
--
-- In order to use the molded crystal in recipes it needs to be burned in the
-- oven. This returns an industrial Mese crystal. The industrial Mese crystal
-- uses the same texture as the default one but the first frame of the cracking
-- animation is added as an impurity effect.

core.register_craftitem(id.industrial_crystal, {
    description = S('Industrial Mese Crystal'),
    inventory_image = 'default_mese_crystal.png^[cracko:1:0'
})

core.register_craft({
    type = 'cooking',
    recipe = id.molded_crystal,
    output = id.industrial_crystal,
    cooktime = 20
})


-- Mese Block
--
-- For storage and building a block can becrafted using industrial Mese
-- crystals, like it is possible with the default Mese crystals.
--
--  ┌───┬───┬───┐
--  │ x │ x │ x │
--  ├───┼───┼───┤
--  │ x │ x │ x │
--  ├───┼───┼───┤
--  │ x │ x │ x │
--  └───┴───┴───┘
--
--  The block also has a visual impurity effect using the crack overlay.

local block_def = table.copy(core.registered_nodes['default:mese_block'])

core.register_node(id.industrial_block, {
    description = S('Industrial Mese Block'),
    tiles = { 'default_mese_block.png^[cracko:2:3' },
    light_source = 1,
    groups = block_def.groups,
    sounds = block_def.sounds,
    mesecons = block_def.mesecons,
    paramtype = 'light'
})

core.register_craft({
    output = id.industrial_block,
    recipe = {
        { id.industrial_crystal, id.industrial_crystal, id.industrial_crystal },
        { id.industrial_crystal, id.industrial_crystal, id.industrial_crystal },
        { id.industrial_crystal, id.industrial_crystal, id.industrial_crystal }
    }
})

core.register_craft({
    output = id.industrial_crystal..' 9',
    type = 'shapeless',
    recipe = { id.industrial_block }
})


-- Mese Shards (Fragments)
--
-- The industrial Mese crystals can be crafted into 5 shards and not fragments,
-- because the industrial crystal can only be shattered into pieces and can
-- not be cleanly fragmented as a gameplay feature. The shards cannot be
-- crafted back into a crystal.
--
-- Without additional mods the shards have no use in *Minetest Game*.

local shard_texture = ('(+base^[cracko:1:2)'):gsub('%+.%w+', {
    ['+base'] = '(default_mese_crystal_fragment.png^[transformR180)'
})

core.register_craftitem(id.industrial_shard, {
    description = S('Industrial Mese Crystal Shard'),
    inventory_image = shard_texture
})

core.register_craft({
    output = id.industrial_shard..' 5',
    type = 'shapeless',
    recipe = { id.industrial_crystal }
})
