local log = imese.log
local register_craft = core.register_craft
local registered_items = table.copy(core.registered_items)
local registered_craftitems = table.copy(core.registered_craftitems)
local get_all_craft_recipes = core.get_all_craft_recipes
local recipe_converter = imese.recipe_converter
local counter = { cooking = 0, normal = 0, fuel = 0, warning = 0 }
local replacements = imese.replacements


-- Log message bases
local log_cooking = 'Registered alternate cooking recipes: '
local log_normal = 'Registered alternate normal crafting recipes: '
local log_fuel = 'Registered alternate fuel recipes: '
local log_warning = 'Not handled recipe method +method of +item'


-- Build the alternate recipe
--
-- This function takes a recipe table as returned by the Luanti API function
-- `get_all_craft_recipes` and replaces all items with the given replacements
-- if the output item is not a replacement itself or one of the own items.
--
-- If the recipe is a normal recipe the normal two-dimensional recipe table is
-- returned. Otherwise the items of the recipe are returned as one-dimensional
-- table (identical with the recipe.items table, but with replaced items).
--
-- If the output is “illegal” as described the function returns nil. This also
-- happens if there was nothing replaced.
--
-- @see core.get_all_craft_recipes
-- @see imese.recipe_converter
-- @param recipe The recipe table as described
-- @return mixed Eitehr a table or nil as described
local build_alternate_recipe = function (recipe)
    local new_items = {}
    local replaced_something = false
    local output = recipe.output:gsub(' .*', '')

    -- Make sure not to create alternative recipes for things to be replaced
    -- or for own items to prevent a loophole for unlimited mese.
    for old,new in pairs(replacements) do if output == old then return end end
    for _,id in pairs(imese.id) do if output == id then return end end

    -- Replace items according to the replacement table and note if something
    -- was replaced
    for id,item in pairs(recipe.items) do
        local new_item = item
        for old,new in pairs(replacements) do
            if item == old then
                new_item = new
                replaced_something = true
            end
        end
        new_items[id] = new_item
    end

    -- If something was replaced create the recipe from the altered items table
    -- and return it.
    if replaced_something == true then
        return recipe_converter(new_items, recipe.width, recipe.method)
    end
end


-- Iterate over the registered items
--
-- This iterates over all items and for each item it gets all crafting recipes.
-- Then it iterates over all crafting recipes (if existent) and processes the
-- recipes so that the alternate recipes are created and the craftitems are
-- registered with that recipes
--
-- @see build_alternate_recipe
-- @see core.register_craft
for id,def in pairs(registered_items) do
    local recipes = get_all_craft_recipes(id) or {}
    if #recipes > 0 then
        for _,recipe in pairs(recipes) do
            local width = recipe.width
            local shape = width == 0 and 'shapeless' or 'shaped'
            local alt_recipe = build_alternate_recipe(recipe)
            local output = recipe.output:gsub(' .*', '')

            -- Reduce the amount to 70 percent of the original amount to
            -- simulate the reduced quality of industrial Mese. The resulting
            -- amount will be the reduced amount (ceiled integer) or 1.
            local amount = tonumber(recipe.output:match(' ([0-9]+)$') or '1')
            local reduced_amount = amount>1 and math.ceil(amount*0.7) or 1

            if alt_recipe then
                if recipe.method == 'cooking' then
                    counter.cooking = counter.cooking + 1
                    register_craft({
                        output = output..' '..reduced_amount,
                        recipe = alt_recipe[1],
                        type = 'cooking',
                        cooktime = width
                    })
                elseif recipe.method == 'normal' then
                    counter.normal = counter.normal + 1
                    register_craft({
                        output = output..' '..reduced_amount,
                        recipe = alt_recipe,
                        type = shape
                    })
                elseif recipe.method == 'fuel' then
                    counter.fuel = counter.fuel + 1
                    register_craft({
                        type = 'fuel',
                        recipe = alt_recipe[0],
                        burntime = reduced_amount
                    })
                else
                    counter.warning = counter.warning + 1
                    log(log_warning:gsub('%+%w+', {
                        ['+method'] = recipe.method,
                        ['+item'] = id
                    }), 'warning')
                end
            end
        end
    end
end


-- Log status for all of the counters
if counter.cooking > 0 then log(log_cooking..counter.cooking) end
if counter.normal > 0 then log(log_normal..counter.normal) end
if counter.fuel > 0 then log(log_fuel..counter.fuel) end
