-- vim:nowrap
--
-- This file contains wide tables to wrapping is disabled.


-- Ladders created with the default wood variants from “Minetest Game”, the
-- steel ladder and the generic wood ladder are left out. Those ladders are
-- handled in a separate registration file.
--
-- @see default.override_ladders.lua


-- Make sure to only run if the currently running game’s ID is `minetest`
-- which is “Minetest Game”.
if ladders_3d.game.id ~= 'minetest' then return end


-- Get the translator
local S = ladders_3d.translator


-- Mass register ladders from a table
--
-- ladders = {
--   short_short_id = { 'My Cool Ladder',  material_definition, 'mymod_my_cool_texture.png',  overlay_opacity },
--   another_id     = { 'My Other Ladder', material_definition, 'mymod_my_other_texture.png', overlay_opacity },
--   -- etc.
-- }
--
-- `material_definition` can either be a single node ID or a table. If it is a
-- single node ID this ID is used as source for the sounds and as crafting
-- material. If it’s a table, the first table entry is used a crafting material
-- and the second entry is used as source for the sounds, etc.
--
-- `overlay_opacity` is number between 0 and 255 to define the opacity of the
-- visual “3D effect mask”.
--
-- @see ladders_3d.register
--
-- @param ladders The table as described
local mass_register = function (ladders)
    local texture_template = '(+base^(+mask^[opacity:+opacity))'
    local image_template = '((+texture)^[mask:+mask)'

    for short_short_id,definition in pairs(ladders) do
        local ladder_name = definition[1]
        local material = definition[2]
        local node_def = core.registered_nodes[definition[2]] or {}

        -- Get node definition
        if type(definition[2]) == 'table' then
            material = definition[2][1]
            node_def = definition[2][2]
        end

        -- Build texture
        local texture = (texture_template:gsub('%+%w+', {
            ['+base'] = '('..definition[3]..')',
            ['+mask'] = 'ladders_3d_default_shape_shading_mask.png',
            ['+opacity'] = definition[4]
        }))

        -- Build image
        local image = (image_template:gsub('%+%w+', {
            ['+texture'] = texture,
            ['+mask'] = 'ladders_3d_default_shape_inventory_mask.png'
        }))

        -- Register ladders
        ladders_3d.register(short_short_id, {
            description = definition[1],
            recipe = {
                { material, '',       material },
                { material, material, material },
                { material, '',       material }
            },
            yield = 7,
            uncraft = material..' 1',
            tiles = { texture },
            sounds = node_def.sounds,
            image = image
        })
    end
end


-- Wooden ladders from default wood nodes
if ladders_3d.option('default_wooden_ladders', true) == true then
    mass_register({
        default_aspen_wood = {  S('Aspen Wood Ladder'),  'default:aspen_wood',  'default_aspen_wood.png^[transformR180',  32 },
        default_pine_wood = {   S('Pine Wood Ladder'),   'default:pine_wood',   'default_pine_wood.png^[transformR180',   32 },
        default_jungle_wood = { S('Jungle Wood Ladder'), 'default:junglewood',  'default_junglewood.png^[transformR180',  32 },
        default_acacia_wood = { S('Acacia Wood Ladder'), 'default:acacia_wood', 'default_acacia_wood.png^[transformR180', 32 },
    })
end


-- Metal ladders from default metal nodes
if ladders_3d.option('default_metal_ladders', true) == true then
    mass_register({
        default_gold = {   S('Golden Ladder'), { 'default:gold_ingot', 'default:gold_block' },     'default_gold_block.png',   32 },
        default_tin = {    S('Tin Ladder'),    { 'default:tin_ingot', 'default:tin_block' },       'default_tin_block.png',    32 },
        default_bronze = { S('Bronze Ladder'), { 'default:bronze_ingot', 'default:bronze_block' }, 'default_bronze_block.png', 64 },
        default_copper = { S('Copper Ladder'), { 'default:copper_ingot', 'default:copper_block' }, 'default_copper_block.png', 32 },
    })
end
