-- Get needed values
local exact_selectionbox = ladders_3d.option('exact_selection_box', false)
local walkable_by_default = ladders_3d.option('walkable_by_default', false)
local game = ladders_3d.game.id
local log = ladders_3d.log


-- Default nodebox resembles the default “Minetest Game” ladder shape
-- but in 3D.
local default_nodebox = {
    type = 'fixed',
    fixed = {
        { -6/16, -8/16, 6/16, -4/16,  8/16, 8/16 }, -- left
        {  4/16, -8/16, 6/16,  6/16,  8/16, 8/16 }, -- right
        { -7/16,  5/16, 6/16,  7/16,  7/16, 8/16 }, -- top
        { -7/16,  1/16, 6/16,  7/16,  3/16, 8/16 }, -- middle top
        { -7/16, -3/16, 6/16,  7/16, -1/16, 8/16 }, -- middle bottom
        { -7/16, -7/16, 6/16,  7/16, -5/16, 8/16 }, -- bottom
    }
}


-- Get the requested selectionbox, if there is none, get the default
-- selectionbox, except the exact selection box should be used.
local selectionbox = function (requested_selectionbox)
    local selectionbox = requested_selectionbox or {
        type = 'fixed',
        fixed = { { -7/16, -8/16, 6/16, 7/16, 8/16, 8/16 } }
    }
    if exact_selectionbox then
        return nil
    else
        return selectionbox
    end
end


-- Register a ladder
--
-- {
--   description = 'My Cool Ladder'
--   recipe = {},
--   yield = 7,
--   uncraft = '',
--   tiles = {},
--   sounds = {},
--   light = 0,
--   groups = {},
--   nodebox = {},
--   selectionbox = {},
--   walkable = false,
--   parameters = {}
-- }
ladders_3d.register = function (short_id, definition)
    local ladder_id = 'ladders_3d:'..short_id

    -- Node parameters
    local parameters = {
        description = definition.description or ladder_id,
        tiles = definition.tiles,
        drawtype = 'nodebox',
        paramtype = 'light',
        node_box = definition.nodebox or default_nodebox,
        selection_box = selectionbox(definition.selectionbox),
        climbable = true,
        walkable = definition.walkable or not walkable_by_default,
        sounds = definition.sounds,
        groups = definition.groups,
        wield_image = definition.image,
        inventory_image = definition.image,
        paramtype2 = 'facedir',
        light_source = definition.light or 0,
    }

    -- Adding extra parameters
    for parameter,value in pairs(definition.parameters or {}) do
        parameters[parameter] = value
    end

    -- Register the node
    core.register_node(':'..ladder_id, parameters)

    -- Register recipe if available
    if definition.recipe then
        core.register_craft({
            output = ladder_id..' '..(definition.yield or 7),
            recipe = definition.recipe
        })
    end

    -- Register “uncrafting recipe” if available
    if definition.uncraft then
        core.register_craft({
            type = 'shapeless',
            output = definition.uncraft,
            recipe = { ladder_id }
        })
    end
end


-- Mass register ladders from a table
--
-- ladders = {
--   short_short_id = {
--     'My Cool Ladder',
--     material_definition,
--     'mymod_my_cool_texture.png',
--     overlay_opacity,
--     groups,
--     nodebox,
--     selectionbox
--   },
--   another_id = {
--     'My Other Ladder',
--     material_definition,
--     'mymod_my_other_texture.png',
--     overlay_opacity,
--     groups,
--     nodebox,
--     selectionbox
--   },
--   -- etc.
-- }
--
-- `material_definition` can either be a single node ID or a table. If it is a
-- single node ID this ID is used as source for the sounds and as crafting
-- material. If it’s a table, the first table entry is used a crafting material
-- and the second entry is used as source for the sounds, etc.
--
-- `overlay_opacity` is number between 0 and 255 to define the opacity of the
-- visual “3D effect mask”.
--
-- @see ladders_3d.register
--
-- @param ladders The table as described
ladders_3d.mass_register = function (ladders)
    log('info', 'Mass registration function invoked.')
    log('info', ' -> Please note that this function is still in development!')

    local texture_template = '(+base^(+mask^[opacity:+opacity))'
    local image_template = '((+texture)^[mask:+mask)'

    for short_short_id,definition in pairs(ladders) do
        local ladder_name = definition[1]
        local material = definition[2]
        local node_def = core.registered_nodes[definition[2]] or {}

        -- Get node definition
        if type(definition[2]) == 'table' then
            material = definition[2][1]
            node_def = definition[2][2]
        end

        -- Build texture
        local texture = (texture_template:gsub('%+%w+', {
            ['+base'] = '('..definition[3]..')',
            ['+mask'] = 'ladders_3d_'..game..'_shading_mask.png',
            ['+opacity'] = definition[4]
        }))

        -- Build image
        local image = (image_template:gsub('%+%w+', {
            ['+texture'] = texture,
            ['+mask'] = 'ladders_3d_'..game..'_inventory_mask.png'
        }))

        -- Register ladders
        ladders_3d.register(short_short_id, {
            description = definition[1],
            recipe = {
                { material, '',       material },
                { material, material, material },
                { material, '',       material }
            },
            yield = 7,
            uncraft = material..' 1',
            tiles = { texture },
            sounds = node_def.sounds,
            image = image,
            groups = definition[5],
            nodebox = definition[6] and {
                type = 'fixed',
                fixed = definition[6]
            } or nil,
            selectionbox = definition[7] and {
                type = 'fixed',
                fixed = definition[7]
            } or nil
        })
    end
end
