local game = ladders_3d.game.id
local modname = ladders_3d.modname


-- Cut a given texture’s edges
--
-- Some node textures (especially the blocks that are often used to create the
-- needed “filler textures” often contain a decorative border that interferes
-- with how the ladders look.
--
-- If only the texture file is provided, the function assumes a 16x16 pixels
-- texture to be cut by 1 pixel (resulting in a 14x14 pixel texture).
--
-- With `in_amount` the amount of pixels to cut can be provided. This does
-- not need any manual calculations, just use the amount you wnat to cut, the
-- function automatically calculates the rest.
--
-- By additionally providing `in_size` you can tell the function what texture
-- size you’re giving it (by default 16x16 pixels are assumed).
--
-- When also giving the function the `out_size`, it will resize the cut texture
-- to that size. If not providing this, it will resize to the `in_size` or 16
-- pixels if not provided.
--
-- Examples:
--
-- `cut_texture('my_cool_texture.png')`
--   * Texture is assumed to be 16x16, is cut to 14x14 (1 pixel
--     cut off on all sides), and then resized to be 16x16 again.
--
-- `cut_texture('my_cool_texture.png', 3, 64, 32)`
--   * Texture will be treated as 64x64 pixels, is cut to 58x58
--     pixels (3 pixels cut off on all sides) and then being
--     resized to 32x32 pixels.
--
-- @param texture   Plain texture file (the file name with suffix)
-- @param in_amount Optional amount to cut off from all corners
-- @param in_size   Optional texture size to cut from
-- @param out_size  Optional target size for the resulting texture
--
-- @return string An encapsulated texture string representing the cut texture
ladders_3d.cut_texture = function (texture, in_amount, in_size, out_size)
    local size = in_size or 16
    local amount = in_amount or 1

    return ('(([combine:+cx+c:+a,+a=(+t))^[resize:+ox+o)'):gsub('%+%w', {
        ['+s'] = size,
        ['+t'] = texture,
        ['+c'] = size - (amount * 2),
        ['+a'] = amount * -1,
        ['+o'] = out_size or size
    })
end


-- Log a status message
--
-- This functions similar to core.log, but automatically prefixes the mod name.
-- If no level is given, `action` is used.
--
-- @param level   One of the supported Luanti log levels
-- @param message The message to log
ladders_3d.log = function (level, message)
    local the_message = message ~= nil and message or level
    local the_level = message == nil and 'action' or level
    core.log(the_level, '[Ladders 3D] '..the_message)
end


-- Get a boolean configuration value or return a default
--
-- This function takes an unprefixed value, automatically prefixes it with the
-- mod name and game and tries to get the boolean value from the configuration
-- (either global or world-specific)
--
-- my_option → ladders_3d_game_my_option
--
-- If `universal` (3rd parameter in function call) is boolean true, then the
-- option is not prefixed by the game.
--
-- my_option → ladders_3d_my_option
--
-- There are two possible locations for the configuration option to be read
-- from: global and world-specific. The global configuration loaded from file
-- that was provided on server start. By default it is `minetest.conf`.
--
-- The file for world-specific configuration needs to be stored in the world’s
-- main directory as `_ladders_3d.conf`. The world-specific parameter always
-- takes precedence of all of the other configuration sources (i.e. the default
-- value and the global configuration).
--
-- If the option is not found in any of the possible configuration locations,
-- the given default value is returned instead.
--
-- @param value     Unprefixed boolean option to get
-- @param default   Default boolean value in case the option is not found
-- @param universal Do not prefix with game name (i.e. independent of game)
--
-- @return mixed The first return value is the value as requested as boolean,
--               or the provided default boolean value. The second return is
--               the full configuration name as string.
ladders_3d.option = function (short_name, default, universal)
    local option_name = modname
        ..(universal and '' or '_'..game)
        ..'_'..short_name

    local global_value = core.settings:get_bool(option_name, default)
    local world_value = ladders_3d.worldconfig:get_bool(option_name)
    local value = world_value == nil and global_value or world_value

    ladders_3d.log('verbose', ('Fetching +t setting: +o = +v'):gsub('%+%w+', {
        ['+t'] = universal and 'mod' or 'specific',
        ['+o'] = option_name,
        ['+v'] = value and 'true' or 'false'
    }))

    return value,option_name
end
