local pl = playtime_limit
local S = playtime_limit.translator
local get_player_by_name = core.get_player_by_name
local c = core.chat_send_player


-- Change playtime counter
--
-- Alter the playtime counter for the given player by the given value. The
-- counter is only altered if the limit is not -1 (disabled).
--
-- @param player_name The name of the player to alter the value for
-- @param value       The value to change the timer by
playtime_limit.change_playtime_counter = function (player_name, value)
    local player = get_player_by_name((player_name or ''))
    if not player then return end

    local meta = player:get_meta()

    if tonumber(meta:get(pl.meta.limit.key)) ~= -1 then
        local current_playtime = meta:get(pl.meta.playtime.key)
        meta:set_string(pl.meta.playtime.key, current_playtime + value)
    end
end


-- Inform a player
--
-- Print a chat message to the player that informs the player about the
-- remaining playtime according to the players individual playtime limit.
--
-- @param validity_table A Validity table
-- @see playtime_limit.validate_player
playtime_limit.inform_player = function (validity_table)
    local player_name = (validity_table or {}).player_name or ''
    local player = get_player_by_name(player_name)
    if not player then return end

    local remaining_time = pl.convert_seconds(validity_table.remaining_time)
    local d = remaining_time.days
    local h = remaining_time.hours
    local m = remaining_time.minutes
    local s = remaining_time.seconds

    c(player_name, S('Your remaining playtime: @1d, @2h, @3m, @4s', d,h,m,s))
end


-- Disconnect a player
--
-- Disconnects the player identified with the validity table. The validity
-- table is used for convenience only. The player will be disconnected
-- regardless of playtime limit.
--
-- @param validity_table A Validity table
-- @see playtime_limit.validate_player
playtime_limit.disconnect_player = function (validity_table)
    local player_name = (validity_table or {}).player_name or ''
    local player = get_player_by_name(player_name)
    if not player then return end

    local reset = pl.convert_seconds(validity_table.session_reset - os.time())
    local d = reset.days
    local h = reset.hours
    local m = reset.minutes
    local s = reset.seconds

    local disconnect_message = table.concat({
        S('Your playtime limit was reached.'),
        S('You’re now being disconnected.'),
        S('You can connect again when your playtime counter is being reset.'),
        '\n\n'..S('Reset will be done in: @1d, @2h, @3m, @4s', d,h,m,s)
    }, ' ')

    core.disconnect_player(player_name, disconnect_message)
end
