local e = core.formspec_escape
local S = void_inventory.translator


-- Return a better placed label
--
-- By default Luanti places labels from the center of the label, this fixes it
-- to place labels from the top left corner like all other elements.
--
-- @param number x    Position on x coordinate
-- @param number y    Position on y coordinate
-- @param string text The text to show as label
--
-- @return string Formspec string for the label
local label = function (x, y, text)
    return 'label['..x..','..(y+0.125)..';'..text..']'
end


-- Return an edit action button
--
-- This function creates a button for the editing actions. The texture is
-- used as icon, and the label is shown as tooltip. If `exit` is `true` then
-- clicking the button closes the inventory.
--
-- @param number  index   Horizontal index
-- @param string  texture Icon to use
-- @param string  name    The name (ID) for that button
-- @param string  label   Tooltip-style label
-- @param boolean exit    Exit the formspec when clicking the button
--
-- @return string Formspec string for the button
local button = function (index, texture, name, label, exit)
    return table.concat({
        'tooltip[+i,0;0.5,0.5;+l]',
        '+type[+i,0;0.5,0.5;+t;+n;;;false]',
    }, ' '):gsub('%+%w+', {
        ['+type'] = exit and 'image_button_exit' or 'image_button',
        ['+i'] = index - 1 == 0 and 0 or (index - 1) * 0.5,
        ['+t'] = '(('..texture..')^[opacity:128)',
        ['+n'] = name,
        ['+l'] = label
    })
end


-- Show an editable field for a meta value
--
-- With this sections the fields are changed. The sections are placed by their
-- index. The sections size is taken into account. The input text fields, the
-- additional information, and the various action buttons are automatically
-- created for the field.
--
-- @param number position Position of the field’s section from top
-- @param string info     Short description of the field’s functionality
-- @param string key      The key that represents this field
-- @param string value    Current value of that field
--
-- @return string Formspec string for the field’s section
local field = function (position, info, key, value)
    return table.concat({
        'container[0.25,+position]',
        '  box[0,0;8.75,1.5;#888a8530]',
        '  style_type[label;font=mono]',
        '  style[data_+key;border=false]',

        label(0.25, 0.25, '+key'),

        '  container[8.625,0.125]',
        button(-2, 'clear.png', 'remove_+key', S('Remove this value')),
        button(-1, 'minimap_btn.png', 'set_+key', S('Set this value')),
        button(-0, 'camera_btn.png', '', info),
        '  container_end[]',

        '  container[0.25,0.75]',
        '    box[0,0;8.25,0.5;#888a8550]',
        '    field_close_on_enter[data_+key;false]',
        '    field[0,0;8.25,0.5;data_+key;;+value]',
        '  container_end[]',

        'container_end[]',
    }, ' '):gsub('%+%w+', {
        ['+position'] = 0.25 + (position - 1) + ((position-1) * 0.75),
        ['+info'] = info,
        ['+key'] = key,
        ['+value'] = (type(value) == 'string' and value or value[key]) or ''
    })
end


-- The meta data fields edit formspec
--
-- @param string    playername Name of the player to show the inventory to
-- @param ItemStack stack      Stack to change the meta data for
-- @param number    scroll     Scroll return position for inventory update
void_inventory.formspecs.edit = function (playername, stack, scroll)

    local id          = stack:get_name()
    local short_desc  = stack:get_short_description() or ''
    local description = stack:get_description() or ''
    local wear        = stack:get_wear()
    local meta        = stack:get_meta()
    local v           = meta:to_table().fields

    return table.concat({
        'formspec_version[8]',
        'size[10,12]',
        'no_prepend[]',
        'bgcolor[#2e3436;both;#2e343550]',
        'listcolors[#babdb650;#eeeeec50;#888a8550]',

        -- Header
        'container[0.25,0.25]',
        '  box[0,0;9.5,1.25;#888a8530]',
        '  style_type[label;font=bold]',
        label(0.25, 0.25, '+name'),
        '  style_type[label;font=mono]',
        label(0.25, 0.75, '+id'),
        '  style_type[label;font=normal]',
        '  tooltip[8.375,0.125;1,1;+description]',
        '  item_image[8.375,0.125;1,1;+id]',
        'container_end[]',

        -- Data fields
        'container[0.25,1.75]',
        '  box[0,0;9.5,9.25;#888a8530]',
        '  scrollbar[9.25,0;0.25,9.25;vertical;data_fields;+scroll]',
        '  scroll_container[0,0;9.25,9.25;data_fields;vertical;;0.25]',
        field(1,  S('Wear value'), 'wear', ''..wear),
        field(2,  S('Pointing range'), 'range', v),
        field(3,  S('Regular description'), 'description', description),
        field(4,  S('Short description'), 'short_description', short_desc),
        field(5,  S('Inventory image'), 'inventory_image', v),
        field(6,  S('Wield image'), 'wield_image', v),
        field(7,  S('Wield overlay'), 'wield_overlay', v),
        field(8,  S('Item color'), 'color', v),
        field(9,  S('Palette index'), 'palette_index', v),
        field(10, S('Displayed counter value'), 'count_meta', v),
        field(11, S('Displayed counter alignment'), 'count_alignment', v),
        '  scroll_container_end[]',
        'container_end[]',

        -- Buttons
        'container[0.25,11.25]',
        '  box[0,0;9.5,0.5;#888a8530]',
        button(18,'inventory_btn.png', 'back', S('Cancel & Back to inventory')),
        button(19, 'clear.png', 'close', S('Cancel & Close'), true),
        'container_end[]',

    }, ' '):gsub('%+%w+', {
        ['+name'] = short_desc,
        ['+description'] = description,
        ['+id'] = id,
        ['+scroll'] = scroll or 0
    })
end
