--[[
Copyright © 2022 LissoBone

Permission is hereby granted, free of charge, to any person obtaining a 
copy of this software and associated documentation files (the 
“Software”), to deal in the Software without restriction, including 
without limitation the rights to use, copy, modify, merge, publish, 
distribute, sublicense, and/or sell copies of the Software, and to 
permit persons to whom the Software is furnished to do so, subject to 
the following conditions:

The above copyright notice and this permission notice shall be included 
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS 
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY 
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
]]

local dot_dist = tonumber(minetest.settings:get("dot_dist")) or 60
local update_interval = tonumber(minetest.settings:get("radar_update_interval")) or 1.5
local play_sounds = tonumber(minetest.settings:get("play_sounds")) or false
local y_pos = tonumber(minetest.settings:get("y_pos")) or 0.45
local x_offset = tonumber(minetest.settings:get("x_offset")) or 200
local y_offset = tonumber(minetest.settings:get("y_offset")) or 0

manhunt = {}
manhunt.hunters = {}
manhunt.huntee = ""
manhunt.manhunt = false
manhunt.huds = {}
manhunt.points = {}

local function hasValue(tbl, value)
   for k, v in ipairs(tbl) do -- iterate table (for sequential tables only)
      if v == value or (type(v) == "table" and hasValue(v, value)) then
	 return true -- Found in this or nested table
      end
   end
   return false -- Not found
end

local function activate_closure(stack, player)
   if stack:get_name() == "manhunt:compass_off" then
      manhunt.huds[player:get_player_name()] = player:hud_add({
	    hud_elem_type = "image",
	    text = "manhunt_radar_background.png",
	    position = {x = 0, y = y_pos},
	    scale = {x = 3, y = 3},
	    alignment = {x = 0, y = 0},
	    offset = {x = x_offset, y = y_offset},
      })

      if manhunt.manhunt == true then
	 manhunt.points[player:get_player_name()] = player:hud_add({
	       hud_elem_type = "image",
	       text = "manhunt_radar_point.png",
	       position = {x = 0, y = y_pos},
	       scale = {x = 3, y = 3},
	       alignment = {x = 0, y = 0},
	       offset = {x = x_offset, y = y_offset},
	 })
      end
      minetest.sound_play("activate_closure", {object = player})
      stack:set_name("manhunt:compass_on")
   elseif stack:get_name() == "manhunt:compass_on" then

      minetest.sound_play("deactivate_closure", {object = player})
      stack:set_name("manhunt:compass_off")
      if manhunt.huds[player:get_player_name()] == nil then
	 return stack
      end
      player:hud_remove(manhunt.huds[player:get_player_name()])
      if manhunt.points[player:get_player_name()] ~= nil then
	 player:hud_remove(manhunt.points[player:get_player_name()])
      end
   end
   return stack
end

local function calc_compass_direction()
   if manhunt.manhunt == false then
      minetest.after(update_interval, calc_compass_direction)
      return
   end

   local huntee = minetest.get_player_by_name(manhunt.huntee)
   if not huntee then
      minetest.after(update_interval, calc_compass_direction)
      return
   end
   
   local huntee_pos = {x = huntee:get_pos().x, z = huntee:get_pos().z}
   for _, huntername in pairs(manhunt.hunters) do
      
      if huntername == "" then
	 minetest.after(update_interval, calc_compass_direction)
	 return
      end
      
      local hunter = minetest.get_player_by_name(huntername)

      if not hunter then
	 minetest.after(update_interval, calc_compass_direction)
	 return
      end
      if manhunt.huds[huntername] == nil then
	 minetest.after(update_interval, calc_compass_direction)
	 return
      end
      
      local inv = hunter:get_inventory()
      
      for _, i in pairs(inv:get_list("main")) do
	 if string.match(i:get_name(), "manhunt:compass_on") then
	    if manhunt.huds[huntername] == nil then
	       return
	    end
	    local hyaw = hunter:get_look_horizontal()
	    local hpos = {x = hunter:get_pos().x, z = hunter:get_pos().z}
	    local dist = math.sqrt((huntee_pos.x-hpos.x)^2 + (huntee_pos.z-hpos.z)^2)
	    local dotpos = {x = 0, y = 0}
	    if dist < 3 then
	       dotpos = {x = 0, y = 0}
	    else
	       dotpos.x = (huntee_pos.x - hpos.x)/dist
	       dotpos.y = (huntee_pos.z - hpos.z)/dist
	       local temporaryx = dotpos.x
	       dotpos.x = ((dotpos.x) * math.cos(hyaw) + (dotpos.y) * math.sin(hyaw))
	       dotpos.y = (-((temporaryx) * math.sin(hyaw)) + ((dotpos.y) * math.cos(hyaw)))
	       temporaryx = nil
	       dotpos.x = (dotpos.x * 60) + x_offset
	       dotpos.y = (-dotpos.y * 60)
	       if manhunt.points[hunter:get_player_name()] == nil then
		  manhunt.points[hunter:get_player_name()] = hunter:hud_add({
	       hud_elem_type = "image",
	       text = "manhunt_radar_point.png",
	       position = {x = 0, y = y_pos},
	       scale = {x = 3, y = 3},
	       alignment = {x = 0, y = 0},
	       offset = {x = x_offset, y = y_offset},
		  })
	       end

	       hunter:hud_change(manhunt.points[hunter:get_player_name()], "offset", dotpos)
	       if play_sounds then
		  minetest.sound_play("radar_update", {object = hunter})
	       end
	       
	       minetest.after(update_interval, calc_compass_direction)
	       return
	    end
	    if play_sounds then
	       minetest.sound_play("radar_alarm", {object = hunter})
	    end
	    
	    hunter:hud_change(manhunt.points[hunter:get_player_name()], "offset", dotpos)
	 else
	    
	    minetest.after(update_interval, calc_compass_direction)
	    return
	 end
      end
      
   end
   minetest.after(update_interval, calc_compass_direction)   
end


minetest.register_craftitem("manhunt:compass_on", {
			       description = "The manhunt compass",
			       inventory_image = "manhunt_compass_on.png",
			       on_use = activate_closure,
			       on_drop = function(itemstack, dropper, pos)
				  newstack = activate_closure(itemstack, dropper)
				  minetest.item_drop(newstack, dropper, pos)
				  return newstack
			       end,
			       groups = {
				  not_in_creative_inventory = 1,
			       },

			       
})

minetest.register_craftitem("manhunt:compass_off", {
			       description = "The manhunt compass",
			       inventory_image = "manhunt_compass_off.png",
			       on_use = activate_closure,
})

minetest.register_craft({
	output = "manhunt:compass_off",
	recipe = {
		{"default:tin_ingot", "default:copper_ingot", "default:tin_ingot"},
		{"default:stick", "default:glass", "default:stick"},
		{"default:stick", "default:steel_ingot", "default:stick"},
	}
})

minetest.register_alias("manhunt_compass", "manhunt:compass_off")

minetest.register_privilege("manhuntadmin", {
			       description = "Can select the huntee in the manhunt and start the game.",
			       give_to_singleplayer = false,
})

minetest.register_chatcommand("sethuntee", {
				 privs = {
				    manhuntadmin = true,
				 },
				 func = function(name, param)
				    if hasValue(minetest.get_connected_players(), minetest.get_player_by_name(param)) then
				       minetest.chat_send_player(name, param .. " has been successfully set as the huntee.")
				       manhunt.huntee = param
				    elseif hasValue(manhunt.hunters, param) then
				       minetest.chat_send_player(name, "A hunter cannot be a huntee at the same time.")
				       
				       return
				    else
				       minetest.chat_send_player(name, param .. " is either not online or already a huntee.")
				       manhunt.huntee = ""
				       return
				    end
				    
				    
end})

minetest.register_chatcommand("removehuntee", {
				 privs = {
				    manhuntadmin = true,
				 },
				 func = function(name, param)
				    if param == manhunt.huntee then
				       manhunt.huntee = ""
				       minetest.chat_send_player(name, param .. " has been removed from the huntee role.")
				    else
				       minetest.chat_send_player(name, param .. " is not a huntee.")
				    end
end})
minetest.register_chatcommand("removehunter", {
				 privs = {
				    manhuntadmin = true,
				 },
				 func = function(name, param)
				    if hasValue(manhunt.hunters, param) then
				       table.remove(manhunt.hunters, param)
				       minetest.chat_send_player(name, param .. " has been successfully removed form the hunter role.")
				       return
				    else
				       minetest.chat_send_player(name, param .. " is either not online or not in the hunter list.")
				       return
				    end
end})

minetest.register_chatcommand("addhunter", {
				 privs = {
				    manhuntadmin = true,
				 },
				 func = function(name, param)
				    if hasValue(manhunt.hunters, param) then
				       minetest.chat_send_player(name, "The specified player is already a hunter.")
				       return
				    elseif not hasValue(minetest.get_connected_players(), minetest.get_player_by_name(param)) then
				       minetest.chat_send_player(name, "No such player found online.")
				       return
				    elseif param == manhunt.huntee then
				       minetest.chat_send_player(name, "A huntee cannot be a hunter at the same time.")
				    end
				    
				    manhunt.hunters[#manhunt.hunters + 1] =  param
				    minetest.chat_send_player(name, "Successfully added " .. param .. " as a hunter.")
end})				    

minetest.register_chatcommand("startmanhunt", {
				 privs = {
				    manhuntadmin = true
				 },
				 func = function(name, param)
				    if #manhunt.hunters ~= 0 and manhunt.huntee ~= "" then
				       manhunt.manhunt = true
				       for _, i in pairs(manhunt.hunters) do
					  if manhunt.points[minetest.get_player_by_name(i)] == nil and manhunt.huds[minetest.get_player_by_name(i)] ~= nil then
					     manhunt.points[i:get_player_name()] = player:hud_add({
						   hud_elem_type = "image",
						   text = "manhunt_radar_point.png",
						   position = {x = 0, y = y_pos},
						   scale = {x = 3, y = 3},
						   alignment = {x = 0, y = 0},
						   offset = {x = offset_x, y = offset_y}
					     })
					  end
				       end
				       minetest.chat_send_all("The manhunt has just started. Make sure to start running, speedrunner!")
				    else
				       minetest.chat_send_player(name, "Not enough participants to start the manhunt")
				    end
end})

minetest.register_chatcommand("stopmanhunt", {
				 privs = {
				    manhuntadmin = true
				 },
				 func = function(name, param)
				    if manhunt.manhunt == true then
				       minetest.chat_send_all("The manhunt has been halted.")
				    else
				       minetest.chat_send_player(name, "The manhunt has not been started yet.")
				    end
end})
minetest.register_chatcommand("hunters", {
				 privs = {
				    manhuntadmin = true,
				 },
				 func = function(name, param)
				    hunters = ""
				    if manhunt.hunters == nil then
				       return
				    end
				    
				    for k, hunter in pairs(manhunt.hunters) do
				       hunters = hunters .. hunter .. "\n"
				    end
				    minetest.chat_send_all(hunters)
end})

minetest.after(1, calc_compass_direction)

minetest.register_on_dieplayer(function(player, reason)
      if player:get_player_name() == manhunt.huntee and manhunt.manhunt == true then
	 manhunt.manhunt = false
	 minetest.chat_send_all("The manhunt has finished. The hunters won.")
      end      
end)
