dofile (minetest.get_modpath("manhunt") .. "/commands.lua")
local has_nomap = minetest.get_modpath("nomap")

local dot_dist = tonumber(minetest.settings:get("dot_dist")) or 60
local update_interval = tonumber(minetest.settings:get("radar_update_interval")) or 1.5
local play_sounds = tonumber(minetest.settings:get("play_sounds")) or false
local y_pos = tonumber(minetest.settings:get("y_pos")) or 0.45
local x_offset = tonumber(minetest.settings:get("x_offset")) or 200
local y_offset = tonumber(minetest.settings:get("y_offset")) or 0

manhunt = {}
manhunt.huds = {}
manhunt.points = {}
manhunt.alts = {}

local function tprint (tbl, indent)
  if not indent then indent = 0 end
  local toprint = string.rep(" ", indent) .. "{\r\n"
  indent = indent + 2 
  for k, v in pairs(tbl) do
    toprint = toprint .. string.rep(" ", indent)
    if (type(k) == "number") then
      toprint = toprint .. "[" .. k .. "] = "
    elseif (type(k) == "string") then
      toprint = toprint  .. k ..  "= "   
    end
    if (type(v) == "number") then
      toprint = toprint .. v .. ",\r\n"
    elseif (type(v) == "string") then
      toprint = toprint .. "\"" .. v .. "\",\r\n"
    elseif (type(v) == "table") then
      toprint = toprint .. tprint(v, indent + 2) .. ",\r\n"
    else
      toprint = toprint .. "\"" .. tostring(v) .. "\",\r\n"
    end
  end
  toprint = toprint .. string.rep(" ", indent-2) .. "}"
  return toprint
end

arena_lib.register_minigame ("manhunt",
			     {icon = "manhunt_compass_on.png",
			      teams = {"Hunters", "Runners"},
			      can_build = true,
			      time_mode = "incremental",
			      load_time = 10,
			      celebration_time = 30,
			      temp_properties = {
				 consent_number = 0,
				 consent_players = {},
			      },
			      player_properties = {
				 consent = false,
			      },
})

----
-- edit this part if u want initial item customization
----

local hunter_items = {}
local runner_items = {}
if has_nomap then
   hunter_items = {"manhunt:compass_off", "nomap:radar_off"}
   runner_items = {"nomap:gps_off"}
else
   hunter_items = {"manhunt:compass_off"}
   -- u get vacuum up your butt
   runner_items = {""}
end

local function hasValue(tbl, value)
   for k, v in ipairs(tbl) do -- iterate table (for sequential tables only)
      if v == value or (type(v) == "table" and hasValue(v, value)) then
	 return true -- Found in this or nested table
      end
   end
   return false -- Not found
end

local function indexOf(array, value)
   for i, v in ipairs(array) do
      if v == value then
	 return i
      end
   end
   return nil
end

local function give_out_items (player, list)
   if not player then return end
   for i, v in ipairs (list) do
      player:get_inventory():set_stack("main", i, v)
      minetest.log (player:get_player_name())
   end
end

local function give_hunter_items (player)
   give_out_items (player, hunter_items)
end

local function give_runner_items (player)
   give_out_items (player, runner_items)
end

local function closest_runner (hunter, runners)
   if not runners then return nil end
   local closest = runners[1]
   for i, v in ipairs (runners) do
      if vector.distance (hunter:get_pos(), v:get_pos()) < vector.distance (hunter:get_pos(), closest:get_pos()) then
	 closest = v
      end
   end
   return closest
end

local function calculate_dotpos (hyaw, hpos, runner_pos)
   local dotpos = {}
   local dist = vector.distance (hpos, runner_pos)
   dotpos.x = (runner_pos.x - hpos.x)/dist
   dotpos.y = (runner_pos.z - hpos.z)/dist
   local temporaryx = dotpos.x
   dotpos.x = ((dotpos.x) * math.cos(hyaw) + (dotpos.y) * math.sin(hyaw))
   dotpos.y = (-((temporaryx) * math.sin(hyaw)) + ((dotpos.y) * math.cos(hyaw)))
   temporaryx = nil
   dotpos.x = (dotpos.x * 60) + x_offset
   dotpos.y = (-dotpos.y * 60)
   return dotpos
end

local function calc_compass_direction(player, arena)
   if not arena.matchID then return nil end
   if not player then return nil end
   
   -- if not arena_lib.is_player_in_arena (player:get_player_name(), arena) then
   --    minetest.chat_send_all ("Not in arena!")
   --    return nil
   -- end
   
   local runners = arena_lib.get_players_in_team (arena, 2, true)
   
   if not runners then return end

   local runner = closest_runner (player, runners)
   if not runner then return end
   local runner_pos = runner:get_pos()
   local huntername = player:get_player_name()
   local inv = player:get_inventory()

   if not inv:contains_item ("main", "manhunt:compass_on") then
      return nil
   end
   
   -- local compass_index = indexOf (inv:get_list ("main"), ItemStack ("manhunt:compass_on"))
   -- if not compass_index then
   --    return nil
   -- end
   
   local hyaw = player:get_look_horizontal()
   local hpos = player:get_pos()
   local dist = vector.distance (hpos, runner_pos)
   local dotpos = {x = x_offset, y = y_offset}
   local alt = ""
   
   if dist > 3 then
      dotpos = calculate_dotpos (hyaw, hpos, runner_pos)
   end

   if math.floor(hpos.y) < math.floor(runner_pos.y) then alt = "manhunt_radar_up.png"
   elseif math.floor(hpos.y) > math.floor(runner_pos.y) then alt = "manhunt_radar_down.png"
   elseif math.floor(hpos.y) == math.floor(runner_pos.y) then alt = "manhunt_radar_eq.png"
   end
   
   if manhunt.points[huntername] == nil then
      manhunt.points[huntername] = player:hud_add({
	    hud_elem_type = "image",
	    text = "manhunt_radar_point.png",
	    position = {x = 0, y = y_pos},
	    scale = {x = 3, y = 3},
	    alignment = {x = 0, y = 0},
	    offset = dotpos
      })
   end
   if manhunt.alts[huntername] == nil then

      manhunt.alts[huntername] = player:hud_add({
	    hud_elem_type = "image",
	    text = "manhunt_radar_eq.png",
	    position = {x = 0, y = y_pos},
	    scale = {x = 3, y = 3},
	    alignment = {x = 0, y = 0},
	    offset = {x = x_offset, y = y_offset},
      })
   end
   player:hud_change(manhunt.points[huntername], "offset", dotpos)
   player:hud_change(manhunt.alts[huntername], "text", alt)

   if play_sounds == true then
      if dist > 3 then
	 minetest.sound_play("radar_alarm", {object = hunter})
      else
	 minetest.sound_play("radar_update", {object = hunter})
      end
   end
   return true
end

local function compass_loop (arena)

   for i, v in ipairs (arena_lib.get_players_in_team (arena, 1, true)) do

      calc_compass_direction (v, arena)
   end

   if not arena.matchID then return end
   minetest.after (update_interval, compass_loop, arena)
end

arena_lib.on_load ("manhunt", function (arena)
		      for i, v in ipairs (arena_lib.get_players_in_team (arena, 1, true)) do
			 give_hunter_items (v)
		      end
		      
		      for i, v in ipairs (arena_lib.get_players_in_team (arena, 2, true)) do
			 give_runner_items (v)
		      end
		      compass_loop (arena)
end)



local function activate_closure(stack, player)
   player_name = player:get_player_name()
   if stack:get_name() == "manhunt:compass_off" then
      manhunt.huds[player_name] = player:hud_add({
	    hud_elem_type = "image",
	    text = "manhunt_radar_background.png",
	    position = {x = 0, y = y_pos},
	    scale = {x = 3, y = 3},
	    alignment = {x = 0, y = 0},
	    offset = {x = x_offset, y = y_offset},
      })

      if manhunt.manhunt == true then
	 manhunt.points[player_name] = player:hud_add({
	       hud_elem_type = "image",
	       text = "manhunt_radar_point.png",
	       position = {x = 0, y = y_pos},
	       scale = {x = 3, y = 3},
	       alignment = {x = 0, y = 0},
	       offset = {x = x_offset, y = y_offset},
	 })
	 manhunt.alts[player_name] = player:hud_add({
	       hud_elem_type = "image",
	       text = "manhunt_radar_eq.png",
	       position = {x = 0, y = y_pos},
	       scale = {x = 3, y = 3},
	       alignment = {x = 0, y = 0},
	       offset = {x = x_offset, y = y_offset},
	 })
      end
      minetest.sound_play("activate_closure", {object = player})
      stack:set_name("manhunt:compass_on")
   elseif stack:get_name() == "manhunt:compass_on" then

      minetest.sound_play("deactivate_closure", {object = player})
      stack:set_name("manhunt:compass_off")
      
      if manhunt.huds[player_name] ~= nil then
	 player:hud_remove(manhunt.huds[player_name])
      end
      
      if manhunt.points[player_name] ~= nil then

	 player:hud_remove(manhunt.points[player_name])
	 manhunt.points[player_name] = nil
      end
      if manhunt.alts[player_name] ~= nil then
	 player:hud_remove(manhunt.alts[player_name])
	 manhunt.alts[player_name] = nil
      end
   end
   return stack
end

arena_lib.on_respawn ("manhunt", function(arena, player_name)
			 local player = minetest.get_player_by_name (player_name)
			 if not player then
			    return
			 end
			 if not arena then return end

			 local manhunt_hunters = arena_lib.get_players_in_team (arena, 1)
			 if hasValue(manhunt_hunters, player_name) then
			    if manhunt.points[player_name] ~= nil then
			       player:hud_remove(manhunt.points[player_name])
			       manhunt.points[player_name] = nil
			    end
			    if manhunt.huds[player_name] ~= nil then
			       player:hud_remove(manhunt.huds[player_name])
			       manhunt.huds[player_name] = nil
			    end
			    if manhunt.alts[player_name] ~= nil then
			       player:hud_remove(manhunt.alts[player_name])
			       manhunt.alts[player_name] = nil
			    end
			    give_hunter_items (player)
			 end
end)



minetest.register_craftitem("manhunt:compass_on", {
			       description = "The manhunt compass",
			       inventory_image = "manhunt_compass_on.png",
			       on_use = activate_closure,
			       on_drop = function(itemstack, dropper, pos)
				  newstack = activate_closure(itemstack, dropper)
				  minetest.item_drop(newstack, dropper, pos)
				  return newstack
			       end,
			       groups = {
				  not_in_creative_inventory = 1,
			       },
			       stack_max = 1,

			       
})

minetest.register_craftitem("manhunt:compass_off", {
			       description = "The manhunt compass",
			       inventory_image = "manhunt_compass_off.png",
			       on_use = activate_closure,
			       stack_max = 1,
})

minetest.register_craft({
      output = "manhunt:compass_off",
      recipe = {
	 {"default:tin_ingot", "default:copper_ingot", "default:tin_ingot"},
	 {"default:stick", "default:glass", "default:stick"},
	 {"default:stick", "default:steel_ingot", "default:stick"},
      }
})

minetest.register_on_joinplayer(function(player, last_login)
      if not player then return end
      if manhunt.huds[player:get_player_name()] ~= nil then return end
      if player:get_inventory():contains_item("main", "manhunt:compass_on") then
	 manhunt.huds[player:get_player_name()] = player:hud_add({
	       hud_elem_type = "image",
	       text = "manhunt_radar_background.png",
	       position = {x = 0, y = y_pos},
	       scale = {x = 3, y = 3},
	       alignment = {x = 0, y = 0},
	       offset = {x = x_offset, y = y_offset}
	 })
      end
end)

arena_lib.on_death("manhunt", function(arena, player, reason)
      local runners = arena_lib.get_players_in_team (arena, 2)
      if hasValue(runners, player) and arena.matchID then
	 arena_lib.remove_player_from_arena(player, 1, "жабобратва")
      end
      if next(arena_lib.get_players_in_team (arena, 2)) == nil then
      	 arena_lib.load_celebration ("manhunt", arena, 1)
      end
end)
