-- constants
local WELCOME_MESSAGE = "Welcome, %s! Type /chat_help <topic>, /chat_tip for a random tip, /chat_found to found a resource or /chat_menu for help."
local UNKNOWN_TOPIC_MESSAGE = "Unknown topic. Try: 'resources', 'building', 'enemies', 'food', 'exploring', 'crafting', 'mining', 'weather', 'biomes', 'minerals', 'iron', 'gold', 'diamonds', 'nether'."
local UNKNOWN_RESOURCE_MESSAGE = "Unknown resource. Try: apple tree, stone, coal, iron, gold, diamond, quartz, glowstone."

-- table for AI responses
local help_responses = {
    resources = "Gather apples from apple trees, find stone in caves, and mine coal underground. Equip a pickaxe!",
    building = "Access the crafting menu by pressing 'i' to create items. Use your hotbar for building.",
    enemies = "Beware of zombies and skeletons. Craft a sword and armor for protection!",
    food = "Hunt cows and pigs for meat, or forage for fruits in different biomes.",
    exploring = "Explore caves and mountains for resources. Bring torches to light your way!",
    crafting = "Use a crafting table for complex items. Check the Luanti wiki for guides.",
    mining = "Dig deep to find diamonds and rare minerals. Upgrade your pickaxe for better results!",
    weather = "Change the weather with /weather [sunny/rainy]. Keep an eye on storms!",
    biomes = "Explore various biomes for unique resources. Don’t miss the deserts and forests!",
    minerals = "Find iron, gold, and diamonds underground. Use a stone pickaxe for iron ore!",
    iron = "Iron ore is found in stone layers. Smelt it in a furnace for iron ingots.",
    gold = "Gold ore is deeper underground. It's less durable than iron but great for decoration.",
    diamonds = "Diamonds are rare and deep in the ground. Use a diamond pickaxe to mine them!",
    nether = "The Nether has unique resources but is dangerous. Bring supplies to survive!",
    nether_resources = "Find Nether Quartz, Soul Sand, and Glowstone in the Nether. Use a diamond pickaxe!"
}

-- list of tips
local random_tips = {
    "Always carry extra tools when exploring!",
    "Cooking food improves its nutritional value!",
    "Build a base to stay safe at night!",
    "Visit villages for valuable resources!",
    "Experiment with crafting for unique items!",
    "Watch the weather; storms can be dangerous!",
    "Use torches to light your surroundings!",
    "Collect resources regularly to stay prepared!",
    "Stay aware of your surroundings to avoid ambushes!",
    "Learn the map layout for shortcuts!",
    "Trade with villagers for unique items!"
}

local last_messages = {}

-- random tip
local function get_random_tip()
    return random_tips[math.random(#random_tips)]
end

-- send a message based on player position
local function check_player_position(player)
    local pos = player:get_pos()
    local player_name = player:get_player_name()
    local message = ""

    -- get time of day
    local time_of_day = minetest.get_timeofday()

    -- determine message based on player's Y position
    if pos.y <= -40 then
        message = "Warning! You are at a height of -40 y. Bring torches for your journey!"
    elseif pos.y <= -30 then
        message = "Be careful! You might encounter lava at these depths."
    elseif pos.y >= 5 and time_of_day >= 0.85 then
        message = "It's night! Build a shelter for your safety."
    elseif pos.y <= -10 then
        message = "You are at an ideal height for mining resources!"
    elseif pos.y >= 80 then
        message = "You are at a high altitude! Build a base for a better view!"
    end

    if message ~= "" and last_messages[player_name] ~= message then
        minetest.chat_send_player(player_name, message)
        last_messages[player_name] = message
    end
end

-- Function to show the help menu
local function show_ai_menu(player)
    local player_name = player:get_player_name()
    local formspec = "size[8,8]label[0.5,0.5;Chat Guide Help Topics]"

    -- Dynamically build help topics
    local index = 1
    for key, value in pairs(help_responses) do
        formspec = formspec .. string.format("label[0.5,%s;%s - %s...]", index * 0.5 + 1, key, value:sub(1, 50))
        index = index + 1
    end

    minetest.show_formspec(player_name, "chat_guide_menu", formspec)
end

-- chat_help
minetest.register_chatcommand("chat_help", { 
    params = "<topic>",
    description = "Ask the Chat Guide for help.", 
    func = function(name, param)
        local response = help_responses[param] or UNKNOWN_TOPIC_MESSAGE
        minetest.chat_send_player(name, response)
        return true
    end
})

-- chat_tip
minetest.register_chatcommand("chat_tip", { 
    description = "Get a random tip from the Chat Guide.",
    func = function(name)
        minetest.chat_send_player(name, get_random_tip())
        return true
    end
})

-- chat_menu
minetest.register_chatcommand("chat_menu", { 
    description = "Open the Chat Guide help menu.",
    func = function(name)
        local player = minetest.get_player_by_name(name)
        if player then
            show_ai_menu(player)
        else
            minetest.chat_send_player(name, "You must be a valid player to access the Chat Guide menu.")
        end
        return true
    end
})

-- welcome message on player join
minetest.register_on_joinplayer(function(player)
    local welcome_message = WELCOME_MESSAGE:format(player:get_player_name())
    minetest.chat_send_player(player:get_player_name(), welcome_message)
end)

-- player position
minetest.register_globalstep(function(dtime)
    for _, player in ipairs(minetest.get_connected_players()) do
        check_player_position(player)
    end
end)

-- calculate the distance between two positions
local function calculate_distance(pos1, pos2)
    return math.sqrt((pos2.x - pos1.x)^2 + (pos2.y - pos1.y)^2 + (pos2.z - pos1.z)^2)
end

-- get the node name for a resource
local function get_node_for_resource(resource_name)
    if not resource_name or type(resource_name) ~= "string" or resource_name == "" then
        return nil
    end

    resource_name = resource_name:lower()

    -- search for the node that matches the resource name
    for node_name, _ in pairs(minetest.registered_nodes) do
        if string.find(node_name:lower(), resource_name) then
            return node_name
        end
    end

    return nil
end

-- chat_found
minetest.register_chatcommand("chat_found", { 
    params = "<resource>",
    description = "Find the nearest resource in the world.",
    func = function(name, resource)
        local player = minetest.get_player_by_name(name)
        if not player then return false end

        if not resource or resource == "" then
            minetest.chat_send_player(name, "Please specify a valid resource to search for.")
            return true
        end

        local actual_node_name = get_node_for_resource(resource)

        if not actual_node_name then
            minetest.chat_send_player(name, string.format("The resource '%s' is not recognized or is not available in the world.", resource))
            return true
        end

        -- player's position
        local player_pos = player:get_pos()
        local closest_distance = math.huge
        local closest_pos = nil

        local radius = 20
        for x = math.floor(player_pos.x - radius), math.floor(player_pos.x + radius) do
            for y = math.floor(player_pos.y - radius), math.floor(player_pos.y + radius) do
                for z = math.floor(player_pos.z - radius), math.floor(player_pos.z + radius) do
                    local node_name = minetest.get_node({x = x, y = y, z = z}).name
                    if node_name == actual_node_name then
                        local distance = calculate_distance(player_pos, {x = x, y = y, z = z})
                        if distance < closest_distance then
                            closest_distance = distance
                            closest_pos = {x = x, y = y, z = z}
                        end
                    end
                end
            end
        end

        -- response to the player
        if closest_pos then
            minetest.chat_send_player(name, string.format("The resource '%s' is %.2f blocks away at coordinates: (%.0f, %.0f, %.0f).",
                                                            resource, closest_distance, closest_pos.x, closest_pos.y, closest_pos.z))
        else
            minetest.chat_send_player(name, string.format("The resource '%s' is not found nearby.", resource))
        end

        return true
    end
})

-- ehi! hai appena trovato un easter egg!