local hyper = ... -- Loads non-global namespace table: hyper

local ourmodpath = core.get_modpath(core.get_current_modname())

local fieldnames = {"Main","Groups","Timer","Functions", "Metadata", "Misc"}
local field_lookup = {
    ["Main"] = 1, ["Groups"] = 2, ["Timer"] = 3,
    ["Functions"] = 4, ["Metadata"] = 5, ["Misc"] = 6 }


local nprops = {} -- properties object
local np = {} -- stores all instances, in case we need them, I guess

local function escape_control_chars(badstring)
    local invalid = {".", "-"}
        -- {"(", ")", ".", "%", "+", "-", "*", "?", "[", "^", "$"}
    for _, char in ipairs(invalid) do
        badstring = badstring:gsub(("%"..char), ("%%%"..char))
    end
    return badstring
end

-- Gets the mod path minus the final folder
-- This is what we're going to delete from a function's full path
local function peel_off_modname(fullpath)
    local tab = fullpath:split("/")
    local out = {}
    for i = 1, #tab - 1 do
        table.insert(out, "/"..tab[i])
    end
    table.insert(out, "/") -- remove leading slash
    return table.concat(out)
end

-- Remove the install path from a function's source, leaving mod and file names
local function get_short_source(modpath, osource)
    modpath = peel_off_modname(modpath)
    local source
    if type(modpath) == "string" then
        local deleter = escape_control_chars(modpath)
        source =
            string.sub(string.gsub(osource, deleter, ""), 2)
    end
    if not source or string.match(source, "//") then
        -- This node's origin mod is in a whole other mod than whatever
        --  added a function to it
        source = source:split("/mods/")[2] -- So use a fallback method
    end
    return source
end

-- Runs over a definition and gets info on all its defined functions
function nprops.parse_node_functions(self)

    -- And this gets the info for each specific function
    local function get_function_data(funcptr)
        local parsed = {}
        local info = debug.getinfo(funcptr)
        local modpath = minetest.get_modpath(self.def.mod_origin) or ""
        parsed.source = get_short_source(modpath, info.source)
        parsed.line = info.linedefined
        return parsed
    end

    self.functions = {} -- clear it and update
    local list = {}
    for key, value in pairs(self.def) do
        if type(value) == "function" then
            table.insert(list, key)
            self.functions[key] = get_function_data(value)
        end
    end
    table.sort(list)
    self.function_list = list

    local fstr = {}
    for i = 1, #list do
        local val = self.functions[list[i]]
        fstr[i] = table.concat(
            {list[i]," : ",val.source," at line ",val.line, "\n" })
    end
    self.function_fstring = table.concat(fstr)
end

function nprops.parse_groups(self)
    local out = { "hypertext[0.5,1;8,5;groups_list;" } local list = {}
    for k, _ in pairs(self.def.groups) do
        table.insert(list, k)
    end
    table.sort(list)
    for i = 1, #list do
        table.insert(out, table.concat({
                             list[i],": ",self.def.groups[list[i]],"\n"}))
    end
    self.group_fstring = table.concat(out)
end

function nprops.parse_timer(self)
    local timer = minetest.get_node_timer(self.pos)
    local started = timer:is_started()
    local hyp = {
            "hypertext[0.5,1;8,5;panel_timer;",
            "Timer active: ",tostring(started),"\n\n",
    }
    local btn = { "field[3.5,6;2,.75;timerset;Timeout;5]",
                  "field_close_on_enter[timerset;false]" }
    local val = self.functions["on_timer"]
    if val then
        table.insert(hyp, table.concat({"Timer function: ",val.source,
                                        " at line ", val.line, "\n"}))
    end
    if started then
        table.insert(hyp, table.concat({
                         "Timeout: ",tostring(timer:get_timeout()),"\n",
                         "Elapsed: ",tostring(timer:get_elapsed()),"\n",
        }))
        -- #TODO: move elapsed to a HUD element that tracks the node at pos
        -- Add a "track" button to main or timer panel to bring it up
        table.insert(btn, table.concat({ -- more buttons
                         "button[1,6;2,.75;timerstop;Stop]",
                         "button[6,6;2,.75;timerfin;Finish]",
        }))
    else
        table.insert(btn, "button[1,6;2,.75;timerstart;Start]")
    end
    table.insert(hyp, "]")
    self.timer_fstring = table.concat(
        {table.concat(hyp),
         table.concat(btn)})
end

-- Top level info getter, reads all node properties into a props object
local function get_node_properties(pos)
    if not pos then return end
    local node = minetest.get_node(pos)
    local props = table.copy(nprops)
    props.name = node.name
    --props.modname = node.name:split(":")[1]
    props.pos = pos
    props.param1 = tostring(node.param1)
    props.param2 = tostring(node.param2)
    props.def = minetest.registered_nodes[node.name]
    props:parse_node_functions()
    props:parse_groups()
    props:parse_timer()
    table.insert(np, props)
    return props
end

local function panel_main(itemstack, props, nodepoint)
    return table.concat({
            "hypertext[0.5,1;8,7;panel_main;",
            "Name: ",props.name,"\n",
            "Desc: ",props.def.short_description
                or props.def.description or "","\n",
            "Mod origin: ",props.def.mod_origin,"\n",
            "Param1: ",props.def.paramtype or "none"," = ",props.param1,"\n",
            "Param2: ",props.def.paramtype2 or "none"," = ",props.param2,"\n",
            "Drawtype: ",props.def.drawtype,"\n",
            "Is_ground_content: ",tostring(props.def.is_ground_content),"\n",
            "Walkable: ",tostring(props.def.walkable),"\n",
            "Buildable_to: ",tostring(props.def.buildable_to),"\n",
            "Connects_to: ",dump(props.def.connects_to),"\n",
            "Drop:",dump(props.def.drop),"\n",
            "]"
    })
end
local function panel_groups(itemstack, props)
    return props.group_fstring.."]"
end
local function panel_timer(itemstack, props)
    return props.timer_fstring
end
local function panel_function(itemstack, props)
    return "hypertext[0.5,1;8,7;panel_function;"..props.function_fstring.."]"
end
local function panel_meta(itemstack, props, player)
    return hyper.panel_meta(core.get_meta(props.pos), player)
end

local panel_misc = loadfile(hyper.modpath.."/misc-node.lua")(hyper)
--[[
local function panel_misc(itemstack, props)
    --return "hypertext[0.5,1;8,7;panel_misc;misc]"
end
]]--

local panelsel = { panel_main, panel_groups, panel_timer,
                   panel_function, panel_meta, panel_misc }
local rmu_users = {}

local function show_formspec(player, itemstack, props)
    if not props then return end
    --local meta = itemstack:get_meta()
    local pname = player:get_player_name()
    if not rmu_users[pname] then rmu_users[pname] = {} end

    local settings = rmu_users[pname]
    rmu_users[pname].props = props
    rmu_users[pname].stack = itemstack

    local nodetab = tonumber(settings.nodetab) or 1
    --local nodepoint = meta:get_string("nodepoint")

    local fstr = {
        "formspec_version[5]",
        "size[9,8]",
        "tabheader[0,0;nodetab;Main,Groups,Timer,Functions,Metadata, Misc;",
        nodetab,";false;true]",
        panelsel[nodetab](itemstack, props, player --, nodepoint
                         )
    }
    minetest.show_formspec(pname, "RMU", table.concat(fstr))
end


minetest.register_on_player_receive_fields(
    function(player, formname, fields)
        if formname ~= "RMU" then return end
        local update = false
        local name = player:get_player_name()
        local settings = rmu_users[name] -- scanner settings
        local props = settings.props -- node properties

        --print("Fields: ",dump(fields))
        --print("Settings: ",dump(settings))

        if fields.nodetab then
            settings.nodetab = fields.nodetab ; update = true
        end
        if ( fieldnames[tonumber(settings.nodetab)] == "Timer"
                 and not fields.quit ) then
            local timer = minetest.get_node_timer(props.pos)
            if fields.timerstop then
                timer:stop()
            elseif fields.timerfin then
                local elapsed = timer:get_timeout()
                timer:start(0, elapsed)
            elseif fields.timerstart or fields.timerset then
                local timeout = tonumber(fields.timerset) or 5
                local elapsed = timer:get_timeout()
                timer:start(timeout, elapsed)
            end
            settings.props:parse_timer() ; update = true
        end
        -- handle recfields on meta tab, etc
        for i = 1, #hyper.tab_recfield do
            if hyper.tab_recfield[i](player, formname, fields, name) then
                update = true
            end
        end
        if update == true then
            show_formspec(player, settings.stack, settings.props)
        end
end)

local cycle = { "hypertrace:pointing_rmu",
                "hypertrace:unpointing_rmu" }

local function item_swap(itemstack)
    local oldname = itemstack:get_name()
    local index = 1
    --local meta = itemstack:get_meta()
    for i = 1, #cycle do
        if cycle[i] == oldname then
            index = i + 1
            break
        end
    end
    if index > #cycle then index = 1 end
    local newstack = ItemStack(cycle[index])
    return newstack
end


-- All-node scanner, that handles unpointable nodes
local update_point, remove_point, unpointed = dofile(ourmodpath.."/hud.lua")

local timer = 0

minetest.register_globalstep(
    function(dtime)
        timer = timer + dtime
        if timer < 0.2 then return end timer = 0

        for _, player in pairs(minetest.get_connected_players()) do
            local pname = player:get_player_name()
            local stack = player:get_wielded_item()
            local itemname = stack:get_name()
            if itemname == "hypertrace:unpointing_rmu" then
                update_point(player, pname, stack)
            elseif unpointed[pname] then
                remove_point(player, pname)
            end
        end
end)

minetest.register_craftitem("hypertrace:pointing_rmu",
    {
        description = "Hypertrace R.M.U. Node Scanner",
        inventory_image = "RMU_scanner.png",
        wield_image = "RMU_scanner.png^[transformFX",
        stack_max = 1,

        on_use = function(itemstack, user, pointed_thing)

            local pos = pointed_thing.under

            show_formspec(user, itemstack, get_node_properties(pos))

        end,

        on_place = function(itemstack, user, pointed_thing)
            return item_swap(itemstack)
        end,

        on_secondary_use = function(itemstack, user, pointed_thing)
            return item_swap(itemstack)
        end,

})

minetest.register_craftitem("hypertrace:unpointing_rmu",
    {
        description = "Hypertrace R.M.U. All-Node Scanner",
        inventory_image = "RMU_scanner.png",
        wield_image = "RMU_scanner.png^[transformFX",
        groups = { not_in_creative_inventory = 1 },
        stack_max = 1,

        on_use = function(itemstack, user, pointed_thing)

            local pname = user:get_player_name()

            if not unpointed[pname] then return end
            local pos = unpointed[pname].pos

            show_formspec(user, itemstack,
                          get_node_properties(pos))

        end,

        on_place = function(itemstack, user, pointed_thing)
            return item_swap(itemstack)
        end,

        on_secondary_use = function(itemstack, user, pointed_thing)
            return item_swap(itemstack)
        end,

})
