local S = core.get_translator(core.get_current_modname());
local F = core.formspec_escape;

function craftoplus.is_minetest() return core.get_game_info().id == "minetest"; end;
function craftoplus.is_mineclonia() return core.get_game_info().id == "mineclonia"; end;

local normal_inits = {
    [tostring(vector.new( 0, 1, 0))] = "top",
    [tostring(vector.new( 0,-1, 0))] = "bottom",
    [tostring(vector.new( 0, 0, 1))] = "front",
    [tostring(vector.new( 0, 0,-1))] = "back",
    [tostring(vector.new( 1, 0, 0))] = "right",
    [tostring(vector.new(-1, 0, 0))] = "left",
};

-- PROPS
if craftoplus.is_minetest() then
    craftoplus.stack_x = 8;
    craftoplus.pref = "mts_";
    craftoplus.slab_pref = "stairs:slab_";
    craftoplus.stair_pref = "stairs:stair_";
    craftoplus.register_door = doors.register;
    craftoplus.register_trapdoor = doors.register_trapdoor;
    craftoplus.textures = {
        water = {
            name = "default_water_source_animated.png",
            backface_culling = false,
            animation = {
                type = "vertical_frames",
                aspect_w = 16, aspect_h = 16,
                length = 2.0,
            },
        },
        lava = {
            name = "default_lava_source_animated.png",
            backface_culling = false,
            animation = {
                type = "vertical_frames",
                aspect_w = 16, aspect_h = 16,
                length = 3.0,
            },
        },
    };
elseif craftoplus.is_mineclonia() then
    craftoplus.stack_x = 9;
    craftoplus.pref = "mcl_";
    craftoplus.slab_pref = "mcl_stairs:slab_";
    craftoplus.stair_pref = "mcl_stairs:stair_";
    craftoplus.register_door = function(name, def) mcl_doors:register_door(name, def); end;
    craftoplus.register_trapdoor = function(name, def) mcl_doors:register_trapdoor(name, def); end;
    craftoplus.textures = {
        water = {
            name="default_water_source_animated.png", 
            animation={
                type="vertical_frames", 
                aspect_w=16, aspect_h=16, 
                length=5.0
            },
        },
        lava = {
            name="default_lava_source_animated.png", 
            animation={
                type="vertical_frames", 
                aspect_w=16, aspect_h=16, 
                length=3.0
            },
        },
    };
end;

-- UPDATE GROUPS
function craftoplus.update_groups(item, groups)
    local groups = core.registered_items[item].groups;
    for name, value in pairs(groups) do
        groups[name] = value;
    end;
    core.override_item(item, { groups = groups });
end;

-- GET INVENTORY
function craftoplus.get_inventory(pos, name, size)
    local meta = core.get_meta(pos);
    local inv = meta:get_inventory();
    if (inv:get_size("main") < 1) then
        inv:set_size(name, size);
    end;
    return meta, inv;
end;

-- SET INVENTORY
function craftoplus.set_inventory(pos, name, size)
    local meta = core.get_meta(pos);
    local inv = meta:get_inventory();
    inv:set_size(name, size);
    return meta, inv;
end;

-- GET ITEM
function craftoplus.get_item(minetest_item, mineclonia_item)
    if core.registered_nodes[minetest_item] or 
        core.registered_items[minetest_item] then
            return minetest_item;
    elseif core.registered_nodes[mineclonia_item] or 
        core.registered_items[mineclonia_item] then
            return mineclonia_item;
    end;
    return nil;
end;

-- INVENTORY FORMSPEC
function craftoplus.inventory_formspec(clicker, id, name, stack, pos, x, y, formspec_only)
    craftoplus.get_inventory(pos, stack, x * y);
    
    local node_pos = pos.x..","..pos.y..","..pos.z;
    local node_inv_bg = "";
    local first_inv_part = "";
    local second_inv_part = "";
    local window_size_gap = 0;
    
    if craftoplus.is_minetest() then
        window_size_gap = 2.5;
        first_inv_part = "list[current_player;main;0.375,"..tostring(y+2.25)..";8,1;]";
        second_inv_part = "list[current_player;main;0.375,"..tostring(y+3.7)..";8,3;8]";
        
    elseif craftoplus.is_mineclonia() then
        window_size_gap = 2.75;
        node_inv_bg = mcl_formspec.get_itemslot_bg_v4(0.375, 0.75, 9, y);
        
        first_inv_part = mcl_formspec.get_itemslot_bg_v4(0.375, y+2.25, 9, 3).."\n";
        first_inv_part = first_inv_part.."list[current_player;main;0.375,"..tostring(y+2.05)..";9,3;9]";
        second_inv_part = mcl_formspec.get_itemslot_bg_v4(0.375, y+6.2, 9, 1).."\n";
        second_inv_part = second_inv_part.."list[current_player;main;0.375,"..tostring(y+6.2)..";9,1;]";
        
    end;
    
    local formspec = table.concat({
        "formspec_version[4]",
        "size["..tostring(x+window_size_gap)..","..tostring(y+7.6).."]",
            
        "label[0.375,0.375;"..F(S(name)).."]", node_inv_bg,
        "list[nodemeta:"..node_pos..";main;0.375,0.75;"..tostring(x)..","..tostring(y)..";]",
            
        "label[0.375,"..tostring(y+1)..".9;"..F(S("Inventory")).."]",
        first_inv_part, second_inv_part,
            
        "listring[nodemeta:"..node_pos..";main]",
        "listring[current_player;main]",
    });
    
    if formspec_only then
        return formspec;
    else
        core.show_formspec(
            clicker:get_player_name(), 
            id.."_"..pos.x.."_"..pos.y.."_"..pos.z, 
            formspec
        );
    end;
end;

-- WOODS
if craftoplus.is_minetest() then
    craftoplus.stick = "group:stick";
    craftoplus.woods = {
        wood = "default:wood", 
        junglewood = "default:junglewood",
        pine_wood = "default:pine_wood",
        acacia_wood = "default:acacia_wood",
        aspen_wood = "default:aspen_wood"
    };
    
elseif craftoplus.is_mineclonia() then
    craftoplus.stick = "mcl_core:stick";
    craftoplus.woods = {
        acacia = "mcl_trees:wood_acacia",
        birch = "mcl_trees:wood_birch",
        oak = "mcl_trees:wood_oak",
        dark_oak = "mcl_trees:wood_dark_oak",
        pale_oak = "mcl_trees:wood_pale_oak",
        spruce = "mcl_trees:wood_spruce",
        jungle = "mcl_trees:wood_jungle",
        bamboo = "mcl_trees:wood_bamboo",
        mangrove = "mcl_trees:wood_mangrove",
        cherry_blossom = "mcl_trees:wood_cherry_blossom",
        warped = "mcl_trees:wood_warped",
        crimson = "mcl_trees:wood_crimson",
    };
end;

-- LOAD NODES
for i, node in pairs(core.registered_nodes) do
    -- LOAD SLABS
    if node.name:find(craftoplus.slab_pref) then
        if (not node.name:find("upside_down")) and 
        (not node.name:find("_top")) and (not node.name:find("_double")) then
            craftoplus.slabs[node.name] = core.registered_nodes[node.name];
        end;
    end;
    
end;

table.sort(craftoplus.slabs);

-- LOAD SOUNDS
if craftoplus.is_minetest() and core.get_modpath('default') then
    craftoplus.sound_wood = default.node_sound_wood_defaults();
    craftoplus.sound_metal = default.node_sound_metal_defaults();
    
elseif craftoplus.is_mineclonia() and core.get_modpath('mcl_sounds') then
    craftoplus.sound_wood = mcl_sounds.node_sound_wood_defaults();
    craftoplus.sound_metal = mcl_sounds.node_sound_metal_defaults();

end;

-- ItemStack
function craftoplus.itemstack(item, itemstack, qnt)
    if core.registered_nodes[item] then
        if qnt then
            return ItemStack(item.." "..tostring(qnt));
        else
            return ItemStack(item);
        end
    else return itemstack or ItemStack() end;
end;

-- DROP ITEMS
function craftoplus.drop_items(pos, stack)
    if stack or (not stack:is_empty()) then
        core.add_item(
            vector.add(
                pos, vector.new(math.random()-0.5, 0, math.random()-0.5)
            ), 
            stack
        );
    end;
end;

function craftoplus.node_drop_items(pos, invs)
    local node = core.get_node(pos);
    invs = invs or {"main"};
    
    if node and node.inventory then
        for i, inv_name in pairs(invs) do
            local list = node.inventory[inv_name];
            if list then
                for i, stack in pairs(list) do
                    craftoplus.drop_items(pos, stack);
                end;
            end;
        end;
    else
        local meta = core.get_meta(pos);
        local inv = meta:get_inventory()
        for i, listname in pairs(invs) do
            for i = 1, inv:get_size(listname) do
                craftoplus.drop_items(pos, inv:get_stack(listname, i));
            end
        end
        meta:from_table();
    end;
end;

-- IS POINTING ABOVE MIDDLE
function craftoplus.is_pointing_above_middle(pointer, pointed_thing)
    local above = pointed_thing.above;

    if not pointer or not pointer:is_player() or not pointed_thing
    or not pointed_thing.under or not above then
        return false
    end

    local fpos = core.pointed_thing_to_face_pos(pointer, pointed_thing).y - above.y;

    return (fpos > 0.05);
end

-- CHECK ON RIGHT CLICK
function craftoplus.check_on_rightclick(itemstack, placer, pointed_thing)
    if pointed_thing and placer and placer:is_player() then
        local pos = pointed_thing.under;
        local node = core.get_node(pos);
        local nodedef = core.registered_nodes[node.name];
        if nodedef.on_rightclick and (not placer:get_player_control().sneak) then
            return nodedef.on_rightclick(pos, node, placer, itemstack, pointed_thing) or itemstack;
        end;
    end;
    return nil;
end;

-- PLACE ITEM
function craftoplus.place_item(itemstack, placer, pointed_thing, real_itemstack)
    real_itemstack = real_itemstack or itemstack;
    local pos = pointed_thing.above;
    local param2 = core.dir_to_facedir(placer:get_look_dir()) % 4;
    core.set_node(pos, {name=itemstack:get_name(), param2=param2});
    core.check_for_falling(pos);
    if not core.is_creative_enabled(placer:get_player_name()) then
        real_itemstack:take_item();
    end;
    return real_itemstack;
end


-- PLACE SWITCHED ITEM
function craftoplus.place_switched_item(item, itemstack, placer, pointed_thing)
    if placer:get_player_control().sneak then
        return craftoplus.place_item(ItemStack(item), placer, pointed_thing, itemstack);
    else
        local node = core.registered_items[itemstack:get_name()];
        if node and node.on_place then
            return node.on_place(itemstack, placer, pointed_thing);
        end;
        return itemstack;
    end;
end