local S = core.get_translator(core.get_current_modname());
local F = core.formspec_escape;
local C = core.colorize;

local storage = core.get_mod_storage();
local is_player_standing = {};

warp_portal = {
    game = core.get_game_info().id,
    mineclonia = "mineclonia",
    minetest = "minetest",
    yaw_map = { [0]=math.pi, [1]=math.pi/2, [2]=0, [3]=-math.pi/2},
    gate_yaw_map = { [0]=0, [1]=-math.pi/2, [2]=math.pi, [3]=math.pi/2},
    anim_close = { x=0, y=17 },
    anim_open = { x=17, y=35 },
};

local button_box = {
    type = "fixed",
    fixed = {
        { -0.3, -0.5, 0.3,  0.3, 0.6, 0.5 },
    },
};

local node_box = {
    type = "fixed",
    fixed = {
        { -1.0, -0.5, -0.3,  1.0, 0.0, 0.5 },
        { -0.3,  0.0,  0.35, 0.3, 1.0, 0.5 },
        { -1.0,  0.0, -0.3, -0.5, 2.0, 0.5 },
        {  1.0,  0.0, -0.3,  0.5, 2.0, 0.5 },
        { -1.0,  2.0, -0.3,  1.0, 2.5, 0.5 },
    },
};

-- IMPORTS
local mod_path = core.get_modpath("warp_portal");

dofile(mod_path.."/utils/node.lua");
dofile(mod_path.."/utils/set_name.lua");
dofile(mod_path.."/utils/teleport.lua");


-- TILES
local function get_tiles(game)
    if game == warp_portal.mineclonia then
        return {
            "default_stone_brick.png", "mcl_blackstone_basalt_side_polished.png",
            "default_obsidian.png", "redstone_redstone_block.png",
            "mcl_deepslate_polished.png", "default_wood.png",
        };
    elseif game == warp_portal.minetest then
        return {
            "default_stone_brick.png", "default_stone_block.png",
            "default_obsidian.png", "default_mese_block.png",
            "default_stone_block.png", "default_wood.png",
        };
    else
        return {};
    end;
end;

-- SOUNDS
local function get_sounds(game)
    if (game == warp_portal.mineclonia) and core.get_modpath("mcl_sounds") then
        return mcl_sounds.node_sound_stone_defaults();
    elseif (game == warp_portal.minetest) and core.get_modpath("default") then
        return default.node_sound_stone_defaults();
    else
        return {};
    end;
end;

local function scroll_open(self)
    
end;

-- SCROLL
core.register_entity("warp_portal:scroll", {
    initial_properties = {
        visual = "mesh",
        visual_size = { x=10, y=10, z=10 },
        textures = { "default_paper.png", "default_wood.png" },
        mesh = "scroll.glb",
        pointable = false,
        physical = false,
    },
    on_activate = function(self, staticdata, dtime_s)
        self.object:set_animation(warp_portal.anim_close, 30, 2, false);
    end,
});

-- GATE
core.register_entity("warp_portal:gate", {
    initial_properties = {
        visual = "mesh",
        visual_size = { x=10, y=10, z=10 },
        textures = { "warp_gate_texture.png" },
        glow = 1,
        use_texture_alpha = true,
        mesh = "portal_gate.obj",
        pointable = false,
        physical = false,
    },
});

-- GROUPS
local function get_groups()
    if (warp_portal.game == warp_portal.minetest) then
        return { cracky=1, level=2 };
    else
        return { pickaxey=4, material_stone=1, unmovable_by_piston=1 };
    end;
end;

-- PORTAL
core.register_node("warp_portal:portal", {
    description = S("Warp Portal"),
    _doc_items_longdesc = S("Warp Portal - allows you to teleport between other warp portals."),
    _doc_items_usagehelp = S(""),
    is_ground_content = false,
    inventory_image = "warp_portal.png",
    tiles = get_tiles(warp_portal.game),
    drawtype = "mesh",
    mesh = "portal.obj",
    drop = "warp_portal:portal",
    collision_box = node_box,
    selection_box = node_box,
    paramtype = "light",
    paramtype2 = "facedir",
    on_place = function(itemstack, user, pointed_thing)
        local pos = pointed_thing.above;
        local param2 = core.dir_to_facedir(user:get_look_dir()) % 4;
        
        local is_clear, center, mask = warp_portal.get_warp_mask(pos, nil, user);
        
        if is_clear then
            core.set_node(pos, {name="warp_portal:portal", param2=param2});
            core.set_node(center, {name="warp_portal:hidden_part", param2=param2});
            
            warp_portal.set_name_formspec(pos, user);
            
            local meta = core.get_meta(pos);
            meta:set_string("yaw", tostring(user:get_look_horizontal()));
            
            for i, l_pos in pairs(mask) do
                core.set_node(l_pos, {name="warp_portal:hidden_part", param2=param2});
            end;
                
            if not core.is_creative_enabled(user:get_player_name()) then
                itemstack:take_item();
            end;
        end;
            
        return itemstack;
    end,
    on_destruct = function(pos)
        local meta = core.get_meta(pos);
        local name = meta:get_string("infotext");
        local yaw = tonumber(meta:get_string("yaw"));
        
        storage:set_string(name, "");
        
        local is_clear, center, mask = warp_portal.get_warp_mask(pos, yaw);
        for i, l_pos in pairs(mask) do
            if (core.get_node(l_pos).name == "warp_portal:hidden_part") then
                core.set_node(l_pos, {name="air"});
            end;
        end;
        
        if (core.get_node(center).name == "warp_portal:hidden_part") or
           (core.get_node(center).name == "warp_portal:button") then
            core.set_node(center, {name="air"});
        end;
            
        local objs = core.get_objects_inside_radius(pos, 1);
        for i, obj in ipairs(objs) do
            if obj then
            if (obj:get_luaentity().name == "warp_portal:scroll") or
                (obj:get_luaentity().name == "warp_portal:gate") then
                    obj:remove();
                end;
            end;
        end;
        
        core.sound_play("turn_off", { pos=pos });
    end,
    groups = get_groups(),
    sounds = get_sounds(warp_portal.game),
    _mcl_blast_resistance = 1000,
    _mcl_hardness = 32,
    pointable = true,
});

-- CHECK STATUS
local function get_status(pos)
    local objs = core.get_objects_inside_radius(pos, 1);
    for i, obj in ipairs(objs) do
        if obj and obj:get_luaentity() then
            if (obj:get_luaentity().name == "warp_portal:gate") then
                return true;
            end;
        end;
    end;
    return false;
end;

-- BUTTON
core.register_node("warp_portal:button", {
    description = "Button",
    paramtype = "light",
    paramtype2 = "facedir",
    drawtype = "airlike",
    groups = { not_in_creative_inventory=1, },
    sunlight_propagates = true,
    walkable = false,
    pointable = true,
    diggable = false,
    buildable_to = false,
    floodable = false,
    drop = "",
    selection_box = button_box,
    on_blast = function() end,
    on_rightclick = function(pos, node, clicker, itemstack, pointed_thing)
        pos = vector.new(pos.x, pos.y-1, pos.z);
        warp_portal.teleport_formspec(pos, clicker, get_status(pos));
        return itemstack;
    end,
    on_punch = function(pos, node, puncher, pointed_thing)
        pos = vector.new(pos.x, pos.y-1, pos.z);
        warp_portal.teleport_formspec(pos, puncher, get_status(pos));
    end,
});

-- ON RECEIVE FIELDS
core.register_on_player_receive_fields(function(player, formname, fields)
    
    if player and (formname:find("warp_portal:teleport_")) then
        local x, y, z = formname:match("warp_portal:teleport_(.-)_(.-)_(.*)");
        local pos = { x=tonumber(x), y=tonumber(y), z=tonumber(z) };
        
        for field in pairs(fields) do
            if field:find("warp_to_") then
                local new_x, new_y, new_z = string.sub(field, 9):match("(.-)|(.-)|(.*)");
                local new_pos = vector.new(tonumber(new_x), tonumber(new_y), tonumber(new_z));
                core.sound_play("warp", { pos=pos });
                player:set_pos(new_pos);
                core.sound_play("warp", { pos=new_pos });
                player:set_yaw(warp_portal.gate_yaw_map[core.get_node(pos).param2%4] or 0);
                core.close_formspec(player:get_player_name(), formname);
            end;
        end;
        
        if fields.portal_turn_on then
            core.add_entity(pos, "warp_portal:gate");
            for i, obj in ipairs(core.get_objects_inside_radius(pos, 1)) do
                if obj and obj:get_luaentity() then
                    local param2 = core.get_node(pos).param2;
                    if (obj:get_luaentity().name == "warp_portal:gate") then
                        obj:set_yaw(warp_portal.gate_yaw_map[param2%4] or 0);
                        core.sound_play("turn_on", { pos=pos });
                    end;
                end;
            end;
            
            local meta = core.get_meta(pos);
            storage:set_string(meta:get_string("infotext"), x.."|"..y.."|"..z);
            warp_portal.teleport_formspec(pos, player, get_status(pos));
        end;
        
        if fields.portal_turn_off then
            local objs = core.get_objects_inside_radius(pos, 1);
            for i, obj in ipairs(objs) do
                if obj and obj:get_luaentity() then
                if (obj:get_luaentity().name == "warp_portal:gate") then
                        core.sound_play("turn_off", { pos=pos });
                        obj:remove();
                    end;
                end;
            end;
            
            local meta = core.get_meta(pos);
            storage:set_string(meta:get_string("infotext"), "off");
            warp_portal.teleport_formspec(pos, player, get_status(pos));
        end;
    end;
    
    if player and (formname:find("warp_portal:set_name_")) then
        local x, y, z = formname:match("warp_portal:set_name_(.-)_(.-)_(.*)");
        local pos = { x=tonumber(x), y=tonumber(y), z=tonumber(z) };
        
        if fields.portal_name_done or fields.portal_name then
            if (fields.portal_name == "") then
                warp_portal.set_name_formspec(
                    pos, player, fields.portal_name,
                    F(C("#FF5555", "Sorry, but you can't use empty name!"))
                );
            elseif (storage:get_string(fields.portal_name) == "") then
                core.chat_send_all(
                    player:get_player_name().." "..S("placed warp portal: ")..fields.portal_name
                );
                
                local meta = core.get_meta(pos);
                meta:set_string("infotext", fields.portal_name);
                
                storage:set_string(fields.portal_name, x.."|"..y.."|"..z);
                
                core.add_entity(pos, "warp_portal:gate");
                core.add_entity(pos, "warp_portal:scroll");
                local objs = core.get_objects_inside_radius(pos, 1);
                for i, obj in ipairs(objs) do
                    if obj and obj:get_luaentity() then
                        core.sound_play("turn_on", { pos=pos });
                        local param2 = core.get_node(pos).param2;
                        if (obj:get_luaentity().name == "warp_portal:scroll") then
                            obj:set_yaw(warp_portal.yaw_map[param2%4] or 0);
                            obj:set_animation(warp_portal.anim_close, 30, 2, false);
                        end;
                        if (obj:get_luaentity().name == "warp_portal:gate") then
                            obj:set_yaw(warp_portal.gate_yaw_map[param2%4] or 0);
                        end;
                    end;
                end;
                
                core.close_formspec(player:get_player_name(), formname);
            else
                warp_portal.set_name_formspec(
                    pos, player, fields.portal_name,
                    F(C("#FF5555", "Sorry, but this name if already taken!"))
                );
            end;
        end;
    end;
end);

-- REGISTER RECIPE
if core.get_game_info().id == warp_portal.mineclonia then
    core.register_craft({
        output = "warp_portal:portal",
        recipe = {
            {"mcl_core:stonebrick", "mcl_end:ender_eye", "mcl_core:stonebrick"},
            {"mcl_core:obsidian", "mcl_redstone_torch:redstoneblock", "mcl_core:obsidian"},
            {
                "mcl_stairs:slab_deepslate_polished", 
                "mcl_lectern:lectern", 
                "mcl_stairs:slab_deepslate_polished"
            },
        },
    })
elseif core.get_game_info().id == warp_portal.minetest then
    core.register_craft({
        output = "warp_portal:portal",
        recipe = {
            {"default:stonebrick", "default:mese", "default:stonebrick"},
            {"default:obsidian", "", "default:obsidian"},
            {"stairs:slab_stone_block", "default:wood", "stairs:slab_stone_block"},
        },
    })
end;

core.register_globalstep(function(dtime)
    for _, player in ipairs(core.get_connected_players()) do
        local pos = player:get_pos();
        
        if (core.get_node(pos).name == "warp_portal:portal") then
            if (is_player_standing[player] == nil) then
                for i, obj in ipairs(core.get_objects_inside_radius(pos, 1)) do
                    if obj and obj:get_luaentity() then
                        if (obj:get_luaentity().name == "warp_portal:scroll") then
                            obj:set_animation(warp_portal.anim_open, 30, 2, false);
                            core.set_node(
                                vector.new(pos.x, pos.y+1, pos.z), 
                                { name="warp_portal:button", param2=core.get_node(pos).param2 }
                            );
                        end;
                    end;
                end;
                is_player_standing[player] = pos;
            end;
        else
            if (is_player_standing[player] ~= nil) then
                for i, obj in ipairs(core.get_objects_inside_radius(is_player_standing[player], 1)) do
                    if obj and obj:get_luaentity() then
                        if (obj:get_luaentity().name == "warp_portal:scroll") then
                            pos = obj:get_pos();
                            obj:set_animation(warp_portal.anim_close, 30, 2, false);
                            core.set_node(
                                vector.new(pos.x, pos.y+1, pos.z), 
                                { name="warp_portal:hidden_part"}
                            );
                        end;
                    end;
                end;
                is_player_standing[player] = nil;
            end;
        end;
    end
end)