-- Beds Mod for Minetest Classic (sfan5 fork)

-- Settings
local player_beds = {}
local is_sp = minetest.is_singleplayer()
local enable_respawn = minetest.settings:get_bool("enable_bed_respawn")
if enable_respawn == nil then enable_respawn = true end

local storage = minetest.get_mod_storage()

-- Load saved beds on mod load
local saved = storage:get_string("player_beds")
if saved and saved ~= "" then
    player_beds = minetest.deserialize(saved) or {}
else
    player_beds = {}
end

-- Save beds function
local function save_player_beds()
    storage:set_string("player_beds", minetest.serialize(player_beds))
end

-- Bed Head Node
minetest.register_node("beds_mod:bed", {
    description = "Bed",
    drawtype = "nodebox",
    tiles = {
        "beds_mod_bed_top.png",   -- top
        "beds_mod_bed_side.png",  -- bottom
        "beds_mod_bed_side.png",  -- right
        "beds_mod_bed_side.png",  -- left
        "beds_mod_bed_side.png",  -- back
        "beds_mod_bed_side.png"   -- front (pillow always at head)
    },
    inventory_image = "beds_mod_bed_inv.png",
    wield_image = "beds_mod_bed_inv.png",
    node_box = {
        type = "fixed",
        fixed = {
            {-0.5, -0.5, -0.5, 0.5, 0, 0.5}
        }
    },
    groups = {oddly_breakable_by_hand = 3, wood = 3},
    paramtype = "light",
    paramtype2 = "facedir", -- rotation enabled
    is_ground_content = false,
    diggable = true,

    -- Place 2-node bed
    on_place = function(itemstack, placer, pointed_thing)
        local pos = pointed_thing.above
        local dir = placer:get_look_dir()
        local facedir = minetest.dir_to_facedir(dir)

        local back = vector.add(pos, minetest.facedir_to_dir(facedir))
        if not minetest.is_protected(pos, placer:get_player_name()) and
           minetest.get_node(back).name == "air" then

            minetest.set_node(pos, {name="beds_mod:bed", param2=facedir})
            minetest.set_node(back, {name="beds_mod:bed_foot", param2=facedir})
            if not minetest.is_creative_enabled(placer:get_player_name()) then
                itemstack:take_item()
            end
        end
        return itemstack
    end,

    -- Right-click to set spawn anytime; sleep only at night
    on_rightclick = function(pos, node, clicker)
        local name = clicker:get_player_name()
        if enable_respawn then
            player_beds[name] = pos
            save_player_beds()  -- Save whenever spawn is set
            minetest.chat_send_player(name, "Spawn point set :)")
        end

        local time = minetest.get_timeofday()
        if time < 0.2 or time > 0.805 then
            minetest.set_timeofday(0.23)
            minetest.chat_send_player(name, "You slept through the night :)")
        else
            minetest.chat_send_player(name, "You can only sleep at night, but spawn point is set :)")
        end
    end,

    -- Break both parts
    after_dig_node = function(pos, oldnode, oldmetadata, digger)
        local dir = minetest.facedir_to_dir(oldnode.param2 or 0)
        local foot_pos = vector.add(pos, dir)
        local foot_node = minetest.get_node(foot_pos)
        if foot_node.name == "beds_mod:bed_foot" then
            minetest.remove_node(foot_pos)
        end
    end
})

-- Bed Foot Node
minetest.register_node("beds_mod:bed_foot", {
    description = "Bed (Foot)",
    drawtype = "nodebox",
    tiles = {
        "beds_mod_bed_pillow.png", -- top
        "beds_mod_bed_side.png",   -- bottom
        "beds_mod_bed_side.png",   -- right
        "beds_mod_bed_side.png",   -- left
        "beds_mod_bed_side.png",   -- back
        "beds_mod_bed_side.png"    -- front (no pillow on foot)
    },
    node_box = {
        type = "fixed",
        fixed = {
            {-0.5, -0.5, -0.5, 0.5, 0, 0.5}
        }
    },
    groups = {oddly_breakable_by_hand = 3, not_in_creative_inventory = 1, wood = 3},
    paramtype = "light",
    paramtype2 = "facedir", -- rotate foot part too
    is_ground_content = false,
    diggable = true,
    drop = "",

    -- Break both parts
    after_dig_node = function(pos, oldnode, oldmetadata, digger)
        local dir = minetest.facedir_to_dir(oldnode.param2 or 0)
        local head_pos = vector.subtract(pos, dir)
        local head_node = minetest.get_node(head_pos)
        if head_node.name == "beds_mod:bed" then
            minetest.remove_node(head_pos)
        end
    end
})

-- Respawn Handler
minetest.register_on_respawnplayer(function(player)
    if not enable_respawn then return false end
    local name = player:get_player_name()
    local pos = player_beds[name]
    if pos then
        player:set_pos({x = pos.x, y = pos.y + 1, z = pos.z})
        minetest.chat_send_player(name, "Respawned at bed spawnpoint :)")
        return true
    end
    return false
end)

-- Items
minetest.register_craftitem("beds_mod:string", {
    description = "String",
    inventory_image = "beds_mod_string.png"
})

minetest.register_node("beds_mod:wool", {
    description = "White Wool",
    tiles = {"beds_mod_wool.png"},
    groups = {snappy = 2, choppy = 2, oddly_breakable_by_hand = 3, wood = 3},
    is_ground_content = false,
    diggable = true
})

-- Crafting
minetest.register_craft({
    output = "beds_mod:string 1",
    recipe = {{"default:rat"}}
})

minetest.register_craft({
    output = "beds_mod:wool",
    recipe = {
        {"beds_mod:string", "beds_mod:string"},
        {"beds_mod:string", "beds_mod:string"}
    }
})

minetest.register_craft({
    output = "beds_mod:bed",
    recipe = {
        {"beds_mod:wool", "beds_mod:wool", "beds_mod:wool"},
        {"default:wood", "default:wood", "default:wood"}
    }
})

-- Ensure breakability (no crashes)
minetest.override_item("beds_mod:bed", {
    diggable = true,
    groups = {oddly_breakable_by_hand = 2, wood = 1}
})
minetest.override_item("beds_mod:bed_foot", {
    diggable = true,
    groups = {oddly_breakable_by_hand = 2, wood = 1}
})

