--[[

	NB More Concrete - Adds more concrete types to NodeCore
	Copyright (C) 2024 NameNotQuality

	This file is part of NB More Concrete.

	NB More Concrete is free software: you can redistribute it and/or modify
	it under the terms of the GNU Affero General Public License as published
	by the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	NB More Concrete is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU Affero General Public License for more details.

	You should have received a copy of the GNU Affero General Public License
	along with NB More Concrete. If not, see <https://www.gnu.org/licenses/>.

--]]



-- Lualocals to become able to use globals
-- LUALOCALS? > ----------------------------
local nodecore, core
    = nodecore, core
-- LUALOCALS! < ----------------------------

-- mn - Mod Name shortcut instead of typing the full name (and easier to rename in forks)
local mn = core.get_current_modname()

-- This program is, once again, licensed under the AGPL-3.0-or-later.
-- Here is a command that you can use to offer your modified source code:
--[[
core.register_chatcommand("nb_more_concrete:source", {
	func = function(name)
		-- Change "(link)" to the URL where the source code is offered (preferably https://).
		core.chat_send_player(name, "The source of this program is offered here: (link)")
		return true
	end
})
--]]



-- Defining each concrete goes in a similar fashion.
-- Here is a function to not repeat the same things:
--
-- I don't want LUAnti letting me forget my C++! <- C but more flexible->onexample("OOP") // object pointer
-- I need to make a C version of this! I still remember! chars for more space saving!
-- void define_the_concrete (char* name, char* desc, char crack, char* ingredient, char dmg, char* sounds,
-- 			    char mcolor[3], char alpha, char bondalpha, char door_scuff, char door_volume,
-- 			    char patopacity, bool patinvert, char* psounds, char* mix, char* mixdesc,
-- 			    char scolor[3], bool regd, char* tiled, char* tilew)
local function define_the_concrete(name, desc, tile, crack, ingredient, crude, washed, dmg, sounds, mcolor, alpha,
                                   bondalpha, door_scuff, door_volume, patopacity, patinvert, psounds, mix, mixdesc,
				   scolor, regd, tiled, tilew)
-- Arg examples:
-- name = "chromete"
-- desc = "Chromete"
-- tile = "nb_more_concrete:chromete.png"
-- crack = 3 (cracky group value)
-- ingredient = "nc_optics:glass_opaque"
-- crude = "nc_optics:glass_crude"
-- washed = "nc_terrain:sand"
-- dmg = 2
-- sounds = "nc_terrain_stony"
-- mcolor = {47, 72, 87}
-- alpha = 128 -- bricks
-- bondalpha = 64 -- bricks
-- door_scuff = 60 -- hinge scuffs on the other side
-- door_volume = 75 -- hinge
-- patopacity = 32 -- pat-pattern
-- patinvert = true -- inverts the shade, here its black, for false its white
-- psounds = "nc_terrain_crunchy" -- p-pliant
-- mix = "chromix"
-- mixdesc = "Chromix"
-- scolor = {37, 62, 77} -- swimcolor
-- regd = true -- register dry (mix)
-- tiled = "^(nc_fire_ash.png^[mask:nc_concrete_mask.png)" -- tile dry (to cat with tile like this: tile .. tiled)
-- tilew ="^(nc_fire_ash.png^(nc_terrain_gravel.png^[opacity:128)^
-- [mask:nc_concrete_mask.png)" -- tile wet, which will cat with the png tile

-- Extra variables so no constant concatinating is required
local node = mn .. ":" .. name -- e.g. mn:chromete




-- Creating concrete is easy.
-- You can do it in 4 steps:
-- Define the node
core.register_node(node, {
	description = desc,
	tiles = {tile},
	groups = {
		cracky = crack,
		-- I was experimenting to find out how can I set a variable called 'name' to 1.
		-- I tried load (name .. "=1") but minetest said a function is required instead.
		-- I tried this and it causes a memory leak nuke and world loading forever!
		-- Should I report this?
		-- load(function () return name .. "=1" end) -- set name as group and set it to 1
		-- But this seems to be the solution
		[name] = 1 -- I think the group gets set anyway, but I think still setting it is ok
		-- silktouch = true -- causes error
	},
	drop_in_place = ingredient,
	crush_damage = dmg,
	sounds = nodecore.sounds(sounds),
	mapcolor = mcolor
})

-- Stone Bricks
nodecore.register_stone_bricks(name, desc,
	tile, -- tile
	alpha,bondalpha, -- alpha and boldalpha numbers for the brick texture
	node, -- made from
	{cracky = crack}, -- normal groups
	{	      -- bonded groups
		cracky = crack+1,
		nc_door_scuff_opacity = door_scuff, -- those white lines on the other side compared to the dark line with the handle
		door_operate_sound_volume = door_volume
	},
	mcolor -- mapcolor
)

-- Register the pliant thing
nodecore.register_concrete_etchable({
		basename = node,
		pattern_opacity = patopacity,
		pattern_invert = patinvert,
		pliant = {
			sounds = nodecore.sounds(psounds),
			drop_in_place = "nc_concrete:" .. mix .. "_wet_source", -- e.g. nc_concrete:chromix_wet_source
			silktouch = false
		}
	})



-- localpref thingy, required for the thing to work and not cause undefined node error
local localpref = "nc_concrete:" .. mn:gsub("^nc_", "") .. "_" -- e.g. nc_concrete:nb_more_concrete_chromete_blank_ply

-- And the actual concrete
nodecore.register_concrete({
		name = mix,
		description = mixdesc,
		-- Register mix can be false for e.g. +charcoal variants
		register_dry = regd,
		craft_mix = regd,
		tile_powder = tile .. tiled,
		tile_wet = tile .. tilew,

		sound = psounds,
		groups_powder = {crumbly = crack},
		swim_color = scolor,
		craft_from_keys = {ingredient},
		craft_from = ingredient,
		to_crude = crude,
		to_washed = washed,
		to_molded = localpref .. name .. "_blank_ply", -- e.g. localpref .. chromete_blank_ply
		mapcolor = mcolor
	})
end

-- local function define_the_concrete(name, desc, crack, ingredient, crude, washed, dmg, sounds, mcolor, alpha, tile,
-- 				      bondalpha, door_scuff, door_volume, patopacity, patinvert, psounds, mix,
-- 				      mixdesc, scolor, regd, tiled, tilew)
--                                    (23 args)
-- Chromete

if (core.settings:get_bool("nb_more_concrete_blue",true)) then
define_the_concrete("chromete", "Chromete", -- name and desc
	mn.."_chromete.png", -- tile
	2, -- cracky
	"nc_optics:glass_opaque", -- ingredient
	"nc_optics:glass_crude", -- crude
	"nc_terrain:sand", -- washed
	2, -- fall damage
	"nc_terrain_stony", -- concrete sound
	{r=47, b=72, g=87}, -- concrete mapcolor
	133, 64, -- brick alpha and bondalpha
	45, 75, -- door scuff opacity and noise
	35, true, -- pattern opacity and invert shade
	"nc_terrain_crunchy", -- mix sounds
	"chromix", "Chromix", -- mix name and desc
	{37, 62, 77}, -- swimcolor
	true, -- register mix/dry
	"^(nc_optics_glass_sparkle.png^[opacity:33)^(nc_fire_ash.png^[mask:nc_concrete_mask.png)", -- tile dry
	"^(nc_optics_glass_sparkle.png^[opacity:26)^(nc_fire_ash.png^(nc_terrain_gravel.png" ..
	"^[opacity:128)^[mask:nc_concrete_mask.png)" -- tile wet
)

-- By default, the concrete is mixable by hand.
-- It may not make much sense by itself, but it
-- becomes an even more apparent issue when the
-- ingredient is a rock-hard glass.
--
-- This method will change the registered recipe
-- toolgroup so the mix can only be done using
-- a stone (or better) mallet.

-- These variables are going to be the array indexes
local label1=0
local label2=0

-- Find the right array with right label
for i=1,#nodecore.registered_recipes do
	if (nodecore.registered_recipes[i].label == "mix chromix") then
		label1=i
		break
	end
end

for i=1,#nodecore.registered_recipes do
	if (nodecore.registered_recipes[i].label == "mix chromix (fail)") then
		label2=i
		break
	end
end

-- Lastly, modify the variables inside the tables

--nodecore.log(dump(nodecore.registered_recipes[label1].toolgroups))
--nodecore.log(dump(nodecore.registered_recipes[label2].toolgroups))

nodecore.registered_recipes[label1].toolgroups.thumpy=3
nodecore.registered_recipes[label2].toolgroups.thumpy=3

--nodecore.log(dump(nodecore.registered_recipes[label1].toolgroups))
--nodecore.log(dump(nodecore.registered_recipes[label2].toolgroups))

end


-- Dark Chromete


if (core.settings:get_bool("nb_more_concrete_darkchromete",true)) then
define_the_concrete("dark_chromete", "Sea Chromete",
	mn.."_chromete.png^[colorize:#000019:180", -- tile
	2, -- cracky
	"nc_optics:glass_opaque", -- ingredient (also used for block drop)
	"nc_optics:glass_crude", -- crude
	"nc_terrain:sand", -- washed
	2, -- fall damage
	"nc_terrain_stony", -- concrete sound
	{r=27, b=52, g=67}, -- mapcolor (unfinished)
	133,64, -- brick alpha/bondalpha
	16,75, -- door scuff opacity and noise
	45,false,-- pattern opacity and invert
	"", -- mix sounds (empty)
	"dark_chromix", "Sea Chromix", -- mix name and desc
	{17, 42, 57}, -- swimcolor
	false, -- register dry mix
	"", -- tile dry (empty)
	"^(nc_optics_glass_sparkle.png^[opacity:26)^(nc_fire_ash.png^(nc_terrain_gravel.png" ..
	"^[opacity:128)^[mask:nc_concrete_mask.png)^[colorize:#000019:70" -- tile wet
)



-- To get the Dark Chromete, one will need to add
-- charcoal to the wet chromete mix in the same
-- fashion as with aggregate.
--
-- NodeCore uses a weird function to do this, but
-- I'm just going to make a craft.
--
-- But is there a good reason why the game takes a
-- different approach?

nodecore.register_craft({
	label = "add charcoal to chromix",
	normal = {y=1},
	nodes = {
		{match = "nc_fire:lump_coal", replace = "air"},
		{match = "nc_concrete:chromix_wet_source", replace = "nc_concrete:dark_chromix_wet_source", y=-1}
	}
})

end


-- Green Concrete


if (core.settings:get_bool("nb_more_concrete_green",true)) then
define_the_concrete("green", "Green Concrete",
	mn.."_green.png", -- tile
	1, -- cracky
	"nc_tree:peat", -- ingredient
	"nc_tree:peat", -- crude
	"nc_tree:peat", -- washed
	1, -- fall dmg
	"nc_terrain_stony", -- sounds
	{r=15, b=15, g=3}, -- mapcolor
	240,120, -- brick alphas
	16,100, -- door scuff opacity and noise
	46, false, -- pattern opacity and invert
	"nc_terrain_crunchy", -- mix sounds
	"green_mix", "Green Aggregate", -- mix name and desc
	{15, 15, 3}, -- swim color
	true, -- define dry
	-- tile dry
	"^(nc_terrain_dirt.png^[opacity:90)^(nc_fire_ash.png^(nc_terrain_dirt.png^[opacity:40)^[mask:nc_concrete_mask.png)",
	-- tile wet
	"^(nc_terrain_dirt.png^[opacity:37)^(nc_fire_ash.png^(nc_terrain_gravel.png^[opacity:128)^[mask:nc_concrete_mask.png)"
)
end


-- Gluestone


if (core.settings:get_bool("nb_more_concrete_orange",true)) then
define_the_concrete("eggstone", "Gluestone",
	"nc_concrete_cloudstone.png^[colorize:#FD8F0A45", -- texture
	1, -- crackyness
	"nc_optics:glass_crude", -- ingredient
	"nc_optics:glass_crude", -- crude
	"nc_optics:glass_crude", -- washed
	1, -- fall dmg
	"nc_terrain_stony", -- sound
	{r=88,g=77,b=61}, -- mapcolor
	120,64, -- brick alphas
	55,100, -- door scuff opacity and noise
	33,true, -- pattern opacity and invert
	"nc_terrain_crunchy", -- mix sounds
	"egg_mix", "Gluey Spackling", -- mix name and desc
	{78, 67, 51}, -- swimcolor
	false, -- define dry
	"", -- tile dry
	"^(nc_fire_ash.png^(nc_terrain_gravel.png^[opacity:128)^[mask:nc_concrete_mask.png)" -- tile wet
	)

-- To get Gluey Spackling, it is going to be required
-- to pummel a wet Spackling with an eggcorn, similarly
-- to sticking an optic in place.


nodecore.register_craft({
	label = "mix eggcorn with spackling",
	action = "pummel",
	wield = {name = "nc_tree:eggcorn"},
	consumewield = 1,
	duration = 1,
	nodes = {
		{match = "nc_concrete:cloudmix_wet_source", replace = "nc_concrete:egg_mix_wet_source"}
	}
})

-- Should I add a setting so instead only right-clicking with the eggcorn is required?
-- Or make that a default?

-- Gluestone 1 (more vibrant Gluestone that you can get by pummeling it with another eggcorn)
--if (core.settings:get_bool()) then
end

