

-- Copyright (C) 2021, 2023 Sandro del Toro

-- This file is part of Bones Minetest Mod.

-- Bones is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as
-- published by the Free Software Foundation, either version 3 of the
-- License, or (at your option) any later version.

-- Bones is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.

-- You should have received a copy of the GNU Affero General Public License
-- along with Bones.  If not, see <https://www.gnu.org/licenses/>.



local S = minetest.get_translator(minetest.get_current_modname())

bones = {}

bones.sounds = {}

if minetest.get_modpath("named_waypoints") then
	local waypoints_def = {
		default_name = "bones",
		default_color = 0xFA364F,
		visibility_volume_radius = 300,
		visibility_volume_height = 300,
		discovery_volume_radius = 0,
	}
	named_waypoints.register_named_waypoints("bones", waypoints_def)
end

if minetest.get_modpath("mcl_sounds") then
   bones.sounds = mcl_sounds
end

if minetest.get_modpath("default") then
   bones.sounds = default
end

local function is_owner(pos, name)
   local owner = minetest.get_meta(pos):get_string("owner")
   if owner == "" or owner == name or minetest.check_player_privs(name, "protection_bypass") then
      return true
   end
   return false
end

local function mcl_fs()
   if minetest.get_modpath("mcl_formspec") then
      return
	 "size[9,9.75]"..
	 "list[current_name;main;0,0.3;9,5;]" ..
	 mcl_formspec.get_itemslot_bg(0,0.3,9,5)..
	 "listring[current_name;main]" ..
	 -- player inv
	 "list[current_player;main;0,5.5;9,3;9]"..
	 mcl_formspec.get_itemslot_bg(0,5.5,9,3)..
	 "list[current_player;main;0,8.74;9,1;]"..
	 mcl_formspec.get_itemslot_bg(0,8.74,9,1)..
	 "listring[current_player;main]"
   elseif minetest.get_modpath("default") then
      return "size[8,9]" ..
	 "list[current_name;main;0,0.3;8,4;]" ..
	 "list[current_player;main;0,4.85;8,1;]" ..
	 "list[current_player;main;0,6.08;8,3;8]" ..
	 "listring[current_name;main]" ..
	 "listring[current_player;main]" ..
	 default.get_hotbar_bg(0,4.85)
   else
      return "size[8,9]" ..
	 "list[current_name;main;0,0.3;8,4;]" ..
	 "list[current_player;main;0,4.85;8,1;]" ..
	 "list[current_player;main;0,6.08;8,3;8]" ..
	 "listring[current_name;main]" ..
	 "listring[current_player;main]"
   end
end

local function remove_bones(pos)
	minetest.remove_node(pos)
	if minetest.get_modpath("named_waypoints") then
		named_waypoints.remove_waypoint("bones", pos)
	end
end

local bones_formspec = mcl_fs()

local share_bones_time = tonumber(minetest.settings:get("share_bones_time")) or 1200
local share_bones_time_early = tonumber(minetest.settings:get("share_bones_time_early")) or share_bones_time / 4

-- legacy alias
minetest.register_alias("mcl_bones:bones", "bones:bones")

local bones_def = {
   description = S("Bones"),
   tiles = {
      "bones_top.png^[transform2",
      "bones_bottom.png",
      "bones_side.png",
      "bones_side.png",
      "bones_rear.png",
      "bones_front.png"
   },
   inventory_image = "bones_front.png",
   wield_image = "bones_front.png",
   paramtype2 = "facedir",
   stack_max = 64,
   is_ground_content = false,
   groups = {dig_immediate = 2},
   sounds = bones.sounds.node_sound_gravel_defaults(),
   drop = "",
   light_source = 8,
   _mcl_blast_resistance = 100000,
   _mcl_hardness = 0.5,
   
   can_dig = function(pos, player)
      local inv = minetest.get_meta(pos):get_inventory()
      local name = ""
      if player then
	 name = player:get_player_name()
      end
      return is_owner(pos, name) and inv:is_empty("main")
   end,
   
   allow_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
      if is_owner(pos, player:get_player_name()) then
	 return count
      end
      return 0
   end,
   
   allow_metadata_inventory_put = function(pos, listname, index, stack, player)
      return 0
   end,
   
   allow_metadata_inventory_take = function(pos, listname, index, stack, player)
	   local name
	   if player then
		   name = player:get_player_name()
	   end
	   if name and is_owner(pos, name) and stack and stack:get_count() then
		   --minetest.chat_send_player(name, S("Punch the node to get all the items"))
		   return stack:get_count()
	   end
	   return 0
   end,
   
   on_metadata_inventory_take = function(pos, listname, index, stack, player)
      local meta = minetest.get_meta(pos)
      if meta:get_inventory():is_empty("main") then
	      --local inv = player:get_inventory()
	      -- if inv:room_for_item("main", {name = "bones:bones"}) then
	      --    inv:add_item("main", {name = "bones:bones"})
	      -- else
	      --    minetest.add_item(pos, "bones:bones")
	      -- end
	      remove_bones(pos)
      end
   end,
   
   on_punch = function(pos, node, player)
      if not is_owner(pos, player:get_player_name()) then
	 return
      end
      
      if minetest.get_meta(pos):get_string("infotext") == "" then
	 return
      end
      
      local inv = minetest.get_meta(pos):get_inventory()
      local player_inv = player:get_inventory()
      local has_space = true
      
      for i = 1, inv:get_size("main") do
	 local stk = inv:get_stack("main", i)
	 if player_inv:room_for_item("main", stk) then
	    inv:set_stack("main", i, nil)
	    player_inv:add_item("main", stk)
	 else
	    has_space = false
	    break
	 end
      end
      
      -- remove bones if player emptied them
      if has_space then
	      -- if player_inv:room_for_item("main", {name = "bones:bones"}) then
	      --    player_inv:add_item("main", {name = "bones:bones"})
	      -- else
	      --    minetest.add_item(pos,"bones:bones")
	      -- end
	      remove_bones(pos)
      end
   end,
   
   on_timer = function(pos, elapsed)
      local meta = minetest.get_meta(pos)
      local time = meta:get_int("time") + elapsed
      if time >= share_bones_time then
	 meta:set_string("infotext", S("@1's old bones", meta:get_string("owner")))
	 meta:set_string("owner", "")
      else
	 meta:set_int("time", time)
	 return true
      end
   end,
   on_blast = function(pos)
	   return nil
   end,
}

if minetest.get_modpath("default") then
   default.set_inventory_action_loggers(bones_def, "bones")
end

minetest.register_node("bones:bones", bones_def)

local function may_replace(pos, player)
   local node_name = minetest.get_node(pos).name
   local node_definition = minetest.registered_nodes[node_name]
   
   -- if the node is unknown, we return false
   if not node_definition then
      return false
   end
   
   -- allow replacing air
   if node_name == "air" then
      return true
   end
   
   -- don't replace nodes inside protections
   if minetest.is_protected(pos, player:get_player_name()) then
      return false
   end
   
   -- allow replacing liquids
   if node_definition.liquidtype ~= "none" then
      return true
   end
   
   -- don't replace filled chests and other nodes that don't allow it
   local can_dig_func = node_definition.can_dig
   if can_dig_func and not can_dig_func(pos, player) then
      return false
   end
   
   -- default to each nodes buildable_to; if a placed block would replace it, why shouldn't bones?
   -- flowers being squished by bones are more realistical than a squished stone, too
   return node_definition.buildable_to
end

local drop = function(pos, itemstack)
   local obj = minetest.add_item(pos, itemstack:take_item(itemstack:get_count()))
   if obj then
      obj:set_velocity({
	    x = math.random(-10, 10) / 9,
	    y = 5,
	    z = math.random(-10, 10) / 9,
      })
   end
end

local player_inventory_lists = { "main", "craft", "armor", "offhand" }
bones.player_inventory_lists = player_inventory_lists

local function is_all_empty(player_inv)
   for _, list_name in ipairs(player_inventory_lists) do
      if not player_inv:is_empty(list_name) then
	 return false
      end
   end
   return true
end

minetest.register_on_dieplayer(function(player)
      local bones_mode = minetest.settings:get("bones_mode") or "bones"
      if bones_mode ~= "bones" and bones_mode ~= "drop" and bones_mode ~= "keep" then
	 bones_mode = "bones"
      end
      
      local bones_position_message = minetest.settings:get_bool("bones_position_message") or true
      local player_name = player:get_player_name()
      local pos = vector.round(player:get_pos())
      local pos_string = minetest.pos_to_string(pos)
      
      -- return if keep inventory set or in creative mode
      if bones_mode == "keep" or minetest.is_creative_enabled(player_name) then
	 minetest.log("action", player_name .. " dies at " .. pos_string ..
		      ". No bones placed")
	 if bones_position_message then
	    minetest.chat_send_player(player_name, S("@1 died at @2.", player_name, pos_string))
	 end
	 return
      end
      
      local player_inv = player:get_inventory()
      if is_all_empty(player_inv) then
	 minetest.log("action", player_name .. " dies at " .. pos_string ..
		      ". No bones placed")
	 if bones_position_message then
	    minetest.chat_send_player(player_name, S("@1 died at @2.", player_name, pos_string))
	 end
	 return
      end
      
      -- check if it's possible to place bones, if not find space near player
      if bones_mode == "bones" and not may_replace(pos, player) then
	 local air = minetest.find_node_near(pos, 1, {"air"})
	 if air then
	    pos = air
	 else
	    bones_mode = "drop"
	 end
      end
      
      if bones_mode == "drop" then
	 for _, list_name in ipairs(player_inventory_lists) do
	    for i = 1, player_inv:get_size(list_name) do
	       drop(pos, player_inv:get_stack(list_name, i))
	    end
	    player_inv:set_list(list_name, {})
	 end
	 drop(pos, ItemStack("bones:bones"))
	 minetest.log("action", player_name .. " dies at " .. pos_string ..
		      ". Inventory dropped")
	 if bones_position_message then
	    minetest.chat_send_player(player_name, S("@1 died at @2, and dropped their inventory.", player_name, pos_string))
	 end
	 return
      end
      
      local param2 = minetest.dir_to_facedir(player:get_look_dir())
      minetest.set_node(pos, {name = "bones:bones", param2 = param2})
      
      minetest.log("action", player_name .. " dies at " .. pos_string ..
		   ". Bones placed")
      if bones_position_message then
	 minetest.chat_send_player(player_name, S("@1 died at @2, and bones were placed.", player_name, pos_string))
      end

      if minetest.get_modpath("named_waypoints") then
	      local waypoint_data = {
		      name = S("@1's bones", player_name),
		      discovered_by = {},
	      }
	      waypoint_data.discovered_by[player_name]=true
	      named_waypoints.add_waypoint("bones", pos, waypoint_data)
      end
      
      local meta = minetest.get_meta(pos)
      local inv = meta:get_inventory()
      if minetest.get_modpath("mcl_inventory") then
	 inv:set_size("main", 9 * 5)
      end
      if minetest.get_modpath("default") then
	 inv:set_size("main", 8 * 4)
      end
      
      for _, list_name in ipairs(player_inventory_lists) do
	 for i = 1, player_inv:get_size(list_name) do
	    local stack = player_inv:get_stack(list_name, i)
	    if inv:room_for_item("main", stack) then
	       inv:add_item("main", stack)
	    else -- no space left
	       drop(pos, stack)
	    end
	 end
	 player_inv:set_list(list_name, {})
      end
      
      meta:set_string("formspec", bones_formspec)
      meta:set_string("owner", player_name)
      
      if share_bones_time ~= 0 then
	 meta:set_string("infotext", S("@1's fresh bones", player_name))
	 
	 if share_bones_time_early == 0 or not minetest.is_protected(pos, player_name) then
	    meta:set_int("time", 0)
	 else
	    meta:set_int("time", (share_bones_time - share_bones_time_early))
	 end
	 
	 minetest.get_node_timer(pos):start(10)
      else
	 meta:set_string("infotext", S("@1's bones", player_name))
      end
end)
