

-- Copyright (C) 2021, 2024 Sandro del Toro

-- This file is part of Emeraldbank Minetest Mod.

-- Emeraldbank is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as
-- published by the Free Software Foundation, either version 3 of the
-- License, or (at your option) any later version.

-- Emeraldbank is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.

-- You should have received a copy of the GNU Affero General Public License
-- along with Emeraldbank.  If not, see <https://www.gnu.org/licenses/>.



local S = core.get_translator(core.get_current_modname())

function emeraldbank.broadcast(player, msg, color, timeout)
	if not player:is_player() then return end
	local name = player:get_player_name()
	if core.get_modpath("notifications_wrapper") then
		notifications.queue({
			playerName = name,
			title = S("EmeraldBank Notification"),
			text = msg,
			--image = "mcl_core_emerald.png",
			timeout = timeout or 5
		})
	elseif core.get_modpath("mcl_title") then
		mcl_title.set(player, "actionbar", {text=msg, color=color or "green"})
	else
		core.chat_send_player(name, msg)
	end
end

function emeraldbank.get_emeralds(name)
	atm.read_account(name)
	return atm.balance[name]
end

-- Accepts a player object or player name as input now
function emeraldbank.set_emeralds(player, num)
    if not player then return false end

    local name
    if type(player) == "string" then
        name = player
    else
        name = player:get_player_name()
    end

    atm.read_account(name)

    -- Check if atm.balance[name] exists
    if atm.balance[name] == nil then
        return false
    end

    if num then
        atm.balance[name] = math.floor(num)
	emeraldbank.broadcast(player, S("Emeralds in Bank: @1", atm.balance[name]), "yellow")
        atm.save_account(name)
        return true
    end
    return false
end

-- Accepts a player object or player name as input now
function emeraldbank.add_emeralds(player, num)
    if not player then return false end

    local name
    if type(player) == "string" then
        name = player
    else
        name = player:get_player_name()
    end

    atm.read_account(name)

    -- Check if atm.balance[name] exists
    if atm.balance[name] == nil then
        return false
    end

    if num then
        atm.balance[name] = math.floor(atm.balance[name] + num)
	emeraldbank.broadcast(player, S("Emeralds in Bank: @1", atm.balance[name]), "yellow")
        atm.save_account(name)
        return true
    end
    return false
end

function emeraldbank.update_accounts()
   for _, player in ipairs(minetest.get_connected_players()) do
      if not player or player.is_fake_player then return end
      local meta = player:get_meta()
      local bankemeralds = meta:get_int("emeraldbank:emerald")
      if bankemeralds > 0 then
	      emeraldbank.add_emeralds(player, bankemeralds)
	      meta:set_int("emeraldbank:emerald", 0)
	      return true
      end
      return false
   end
end

function emeraldbank.transfer_emeralds(player1, player2, num)
    -- Determine player names based on whether input is a string or player object
    local name1 = type(player1) == "string" and player1 or player1:get_player_name()
    local name2 = type(player2) == "string" and player2 or player2:get_player_name()

    atm.read_account(name1)
    local bankemeralds1 = atm.balance[name1]

    if num > 0 then
        if bankemeralds1 and bankemeralds1 >= num then
            core.chat_send_player(name1, S("Pay Successfully! You have transferred @1 Emeralds.", num))
            core.chat_send_player(name2, S("Pay Successfully! You've gotten @1 Emeralds.", num))

            local msg = S("@1 has transferred @2 emeralds to @3", name1, num, name2)
            -- Integration with external mods
            if core.get_modpath("irc") then
                irc.say(msg)
            end
            if core.get_modpath("yl_matterbridge") then
                yl_matterbridge.send_to_bridge("EMERALDBANK", msg)
            end
            if core.get_modpath("beerchat") then
                beerchat.on_channel_message(beerchat.main_channel_name, "EMERALDBANK", msg)
            end

            emeraldbank.add_emeralds(name1, -num)  -- Now using name1 instead of player1
            emeraldbank.add_emeralds(name2, num)   -- Now using name2 instead of player2

            -- Sound effect for player2
            if type(player2) ~= "string" then
                core.sound_play("cash", {
                    to_player = name2,
                    gain = 1.0,
                    fade = 0.0,
                    pitch = 1.0,
                })
            end
        else
            core.chat_send_player(name1, S("Not enough Emeralds in your account"))
        end
    else
        core.chat_send_player(name1, S("Invalid pay"))
    end
end



-- Shops API

function emeraldbank.inv_emeralds_to_stonks(pos)
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	local stonks = meta:get_int("stonks")
	local fancy_inv = inv:get_list("main")
	if not fancy_inv then return end
	local has_emerald = inv:contains_item("main", "mcl_core:emerald 1", true)
	if has_emerald then
		meta:set_int("stonks", stonks+1)
		inv:remove_item("main", "mcl_core:emerald 1")
		emeraldbank.inv_emeralds_to_stonks(pos)
		return true
	else
		return false
	end
end

function emeraldbank.get_stonks(pos)
	local meta = core.get_meta(pos)
	local owner = meta:get_string("owner")
	local player = core.get_player_by_name(owner)
	local is_online = core.player_exists(owner)
	emeraldbank.inv_emeralds_to_stonks(pos)
	local stonks = meta:get_int("stonks")
	if not player or player.is_fake_player then return end
	if is_online and stonks > 0 then
		core.sound_play("cash", {
			to_player = owner,
			gain = 1.0,
			fade = 0.0,
			pitch = 1.0,
		})
		emeraldbank.add_emeralds(player, stonks)
		meta:set_int("stonks", 0)
		core.chat_send_player(owner, S("You've earned @1 Emeralds with your shops.", stonks))
	end
end
