

-- Copyright (C) 2021, 2024 Sandro del Toro

-- This file is part of Emeraldbank Minetest Mod.

-- Emeraldbank is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as
-- published by the Free Software Foundation, either version 3 of the
-- License, or (at your option) any later version.

-- Emeraldbank is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.

-- You should have received a copy of the GNU Affero General Public License
-- along with Emeraldbank.  If not, see <https://www.gnu.org/licenses/>.



local S = core.get_translator(core.get_current_modname())


-- user /pay chat command
core.register_chatcommand("pay", {
    params = "<player> <num>",
    description = S("Pay money to other player. Transfer your emeralds to another bank account."),
    func = function(name, param)
        local name2, stringnum = param:match("([^ ]+) (.+)")
        local num = tonumber(stringnum)
        if name2 and num then
            -- Check if the balance exists for the target player
            atm.read_account(name2)
            if atm.balance[name2] == nil then
                core.chat_send_player(name, S("Player @1 does not exist or has no account.", name2))
                return false
            end

            -- Use strings for player names in the transfer function
            return emeraldbank.transfer_emeralds(name, name2, num)
        else
            core.chat_send_player(name, S("Invalid input. Please specify a player and a number."))
            return false
        end
    end
})


-- user /money chat command
core.register_chatcommand("money", {
    description = S("Return your emeralds in your bank account. Or if have server priv, other players too"),
    params = "[player]",
    func = function(name, param)
        -- Determine the target player: either the command issuer or the specified player
        local target_player = param ~= "" and minetest.check_player_privs(name, {server=true}) and param or name

        -- Check and return the emerald balance
        local emeralds = emeraldbank.get_emeralds(target_player)
        if emeralds then
            if target_player == name then
                minetest.chat_send_player(name, S("Emeralds in Bank: @1", emeralds))
            else
                minetest.chat_send_player(name, S("@1's Emeralds in Bank: @2", target_player, emeralds))
            end
            return true
        end
        return false
    end
})


-- admin chat command
core.register_chatcommand("emeralds", {
    params = "<player> <num>",
    description = S("Admin Command! Add player emeralds in bank account, also can use negative numbers"),
    privs = {server=true},
    func = function(name, param)
        local playername, stringnum = param:match("([^ ]+) (.+)")
        local num = tonumber(stringnum)
        if playername and num then
            local success = emeraldbank.add_emeralds(playername, num)
            if success then
                atm.read_account(playername)
                minetest.chat_send_player(name, S("@1 has now @2 emeralds in bank account",
                playername, atm.balance[playername]))
                return true
            else
                minetest.chat_send_player(name, S("Player @1 not found or not online.", playername))
                return false
            end
        else
            -- Notify the command issuer that the input is invalid
            minetest.chat_send_player(name, S("Invalid input. Please specify a player and a number."))
            return false
        end
    end
})
