

-- Copyright (C) 2021, 2026 Sandro del Toro

-- This file is part of Emeraldbank Minetest Mod.

-- Emeraldbank is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as
-- published by the Free Software Foundation, either version 3 of the
-- License, or (at your option) any later version.

-- Emeraldbank is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.

-- You should have received a copy of the GNU Affero General Public License
-- along with Emeraldbank.  If not, see <https://www.gnu.org/licenses/>.



local S = core.get_translator(core.get_current_modname())


-- user /pay chat command
core.register_chatcommand("pay", {
	params = "<player> <num> [desc]",
	description = S("Pay money to other player. Transfer your emeralds to another bank account."),
	func = function(name, param)
		local target, amount_s, desc = param:match("^(%S+)%s+(%d+)%s*(.*)$")
		local num = amount_s and tonumber(amount_s) or nil
		if not target or not num then
			core.chat_send_player(name, S("Invalid input. Please specify a player and a number."))
			return false
		end

		-- debe ejecutarse in-game (necesitamos PlayerRef para abrir el formspec)
		local player = core.get_player_by_name(name)
		if not player then
			core.chat_send_player(name, S("This command must be run in-game to confirm the transfer."))
			return false
		end

		-- existencia en auth (offline permitido)
		if not core.player_exists(target) then
			core.chat_send_player(name, S("Player @1 does not exist.", target))
			return false
		end
		
		-- comprobar saldo antes de mostrar confirmación
		atm.read_account(name)
		if atm.balance[name] == nil then
			core.chat_send_player(name, S("Your bank account is not available."))
			return false
		end
		
		if atm.balance[name] < num then
			core.chat_send_player(name, S("Not enough Emeralds in your account"))
			return false
		end
		
		-- preparar transferencia pendiente (modo externo: no vuelve al terminal WT)
		local wdesc = desc ~= "" and desc or S("/pay command")
		atm.pending_transfers[name] = {
			to = target,
			sum = math.floor(num),
			desc = wdesc,
			extern = true,
			kind = "pay",
			mail = false,
			sound = true,
			notify = true,
		}
		
		-- mostrar resumen y confirmar
		atm.showform_wtconf(player, target, tostring(math.floor(num)), wdesc)
		return true
	end
})


-- user /money chat command
core.register_chatcommand("money", {
    description = S("Return your emeralds in your bank account. Or if have server priv, other players too"),
    params = "[player]",
    func = function(name, param)
        -- Determine the target player: either the command issuer or the specified player
        local target_player = param ~= "" and core.check_player_privs(name, {server=true}) and param or name

        -- Check and return the emerald balance
        local emeralds = emeraldbank.get_emeralds(target_player)
        if emeralds then
            if target_player == name then
                core.chat_send_player(name, S("Emeralds in Bank: @1", emeralds))
            else
                core.chat_send_player(name, S("@1's Emeralds in Bank: @2", target_player, emeralds))
            end
            return true
        end
        return false
    end
})


-- admin chat command
core.register_chatcommand("emeralds", {
    params = "<player> <num>",
    description = S("Admin Command! Add player emeralds in bank account, also can use negative numbers"),
    privs = {server=true},
    func = function(name, param)
        local playername, stringnum = param:match("([^ ]+) (.+)")
        local num = tonumber(stringnum)
        if playername and num then
            local success = emeraldbank.add_emeralds(playername, num)
            if success then
                atm.read_account(playername)
                core.chat_send_player(name, S("@1 has now @2 emeralds in bank account",
                playername, atm.balance[playername]))
                return true
            else
                core.chat_send_player(name, S("Player @1 not found or not online.", playername))
                return false
            end
        else
            -- Notify the command issuer that the input is invalid
            core.chat_send_player(name, S("Invalid input. Please specify a player and a number."))
            return false
        end
    end
})
