

-- Copyright (C) 2021, 2026 Sandro del Toro

-- This file is part of Emeraldbank Minetest Mod.

-- Emeraldbank is free software: you can redistribute it and/or modify
-- it under the terms of the GNU Affero General Public License as
-- published by the Free Software Foundation, either version 3 of the
-- License, or (at your option) any later version.

-- Emeraldbank is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU Affero General Public License for more details.

-- You should have received a copy of the GNU Affero General Public License
-- along with Emeraldbank.  If not, see <https://www.gnu.org/licenses/>.



local S = core.get_translator(core.get_current_modname())

function emeraldbank.broadcast(player, msg, color, timeout)
	if not (player and type(player) ~= "string" and player:is_player()) then
		return
	end
	local name = player:get_player_name()
	if core.get_modpath("notifications_wrapper") then
		notifications.queue({
			playerName = name,
			title = S("EmeraldBank Notification"),
			text = msg,
			--image = "mcl_core_emerald.png",
			timeout = timeout or 5
		})
	elseif core.get_modpath("mcl_title") then
		mcl_title.set(player, "actionbar", {text=msg, color=color or "green"})
	else
		core.chat_send_player(name, msg)
	end
end

function emeraldbank.get_emeralds(name)
	atm.read_account(name)
	return atm.balance[name]
end

-- Accepts a player object or player name as input now
function emeraldbank.set_emeralds(player, num)
	if not player then return false end
	
	local name
	if type(player) == "string" then
		name = player
	else
		name = player:get_player_name()
	end
	
	atm.read_account(name)
	
	-- Check if atm.balance[name] exists
	if atm.balance[name] == nil then
		return false
	end
	
	if num then
		atm.balance[name] = math.floor(num)
		emeraldbank.broadcast(player, S("Emeralds in Bank: @1", atm.balance[name]), "yellow")
		atm.save_account(name)
		return true
	end
	return false
end

-- Accepts a player object or player name as input now
function emeraldbank.add_emeralds(player, num)
	if not player then return false end
	
	local name
	if type(player) == "string" then
		name = player
	else
		name = player:get_player_name()
	end
	
	atm.read_account(name)
	
	-- Check if atm.balance[name] exists
	if atm.balance[name] == nil then
		return false
	end
	
	if num then
		atm.balance[name] = math.floor(atm.balance[name] + num)
		emeraldbank.broadcast(player, S("Emeralds in Bank: @1", atm.balance[name]), "yellow")
		atm.save_account(name)
		return true
	end
	return false
end

function emeraldbank.update_accounts(player)
	if not player or player.is_fake_player or not player:is_player() then return end
	
	local meta = player:get_meta()
	local bankemeralds = meta:get_int("emeraldbank:emerald")
	if bankemeralds > 0 then
		emeraldbank.add_emeralds(player, bankemeralds)
		meta:set_int("emeraldbank:emerald", 0)
	end
end

local function _name_of(p)
	if p == nil then return nil end
	if type(p) == "string" then return p end
	if type(p) == "userdata" and p.is_player and p:is_player() then
		return p:get_player_name()
	end
	return nil
end

local function _ensure_tx_list(name)
	atm.read_transaction(name)
	if not atm.completed_transactions[name] then
		atm.completed_transactions[name] = {}
	end
end

local function _push_tx(name, entry)
	_ensure_tx_list(name)
	table.insert(atm.completed_transactions[name], entry)
	atm.write_transaction(name)
end

-- Motor único de transferencias (API pública)
-- from: string|PlayerRef|nil   (nil => SYSTEM)
-- to:   string|PlayerRef
-- amount: número positivo (se fuerza a int)
-- desc: string (opcional)
-- opts: { kind="pay|wire|shops|...", mail=true/false, notify=true/false, sound=true/false, callback=function(entry) end }
function emeraldbank.transfer(from, to, amount, desc, opts)
	opts = opts or {}
	local from_name = _name_of(from)
	local to_name = _name_of(to)
	if type(to_name) ~= "string" or to_name == "" then
		return false, "bad_recipient"
	end
	if not core.player_exists(to_name) then
		return false, "recipient_not_in_auth"
	end
	if type(amount) ~= "number" then
		return false, "bad_amount"
	end
	amount = math.floor(amount)
	if amount <= 0 then
		return false, "amount_must_be_positive"
	end
	atm.read_account(to_name)
	if atm.balance[to_name] == nil then
		return false, "recipient_no_account"
	end
	if from_name then
		atm.read_account(from_name)
		if atm.balance[from_name] == nil then
			return false, "sender_no_account"
		end
		if atm.balance[from_name] < amount then
			return false, "insufficient_funds"
		end
	end
	-- mover saldo
	if from_name then
		atm.balance[from_name] = atm.balance[from_name] - amount
		atm.save_account(from_name)
	end
	atm.balance[to_name] = atm.balance[to_name] + amount
	atm.save_account(to_name)
	local entry = {
		date = os.date("%d/%m/%Y"),
		from = from_name or opts.from_label or "SYSTEM",
		to = to_name,
		sum = amount,
		desc = desc or "",
		kind = opts.kind or "transfer",
	}
	-- log para ambos (si hay emisor)
	_push_tx(to_name, entry)
	if from_name and from_name ~= to_name then
		_push_tx(from_name, entry)
	end
	-- mail opcional (por defecto: solo si opts.mail == true)
	if opts.mail == true and core.get_modpath("mail") and mail.send then
		mail.send({
			from = "Emerald Bank",
			to = to_name,
			subject = S("Payment of @1 completed", amount),
			body = S("Transfer details:\n\nFrom: @1\nTo: @2\nAmount: @3\nDescription:\n@4",
				entry.from, entry.to, entry.sum, entry.desc)
		})
	end
	-- notify/sound opcional
	if opts.notify == true then
		if from_name and core.get_player_by_name(from_name) then
			core.chat_send_player(from_name, S("Transfer completed: @1 -> @2 (@3)", entry.from, entry.to, entry.sum))
		end
		if core.get_player_by_name(to_name) then
			core.chat_send_player(to_name, S("You received @1 emeralds from @2", entry.sum, entry.from))
		end
	end
	if opts.sound == true and core.get_player_by_name(to_name) then
		core.sound_play("cash", { to_player = to_name, gain = 1.0 })
	end
	if type(opts.callback) == "function" then
		opts.callback(entry)
	end
	return true
end


-- Shops API

local function inv_emeralds_to_stonks(pos)
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	if not inv:get_list("main") then return 0 end
	local moved = 0
	while inv:contains_item("main", "mcl_core:emerald 1", true) do
		inv:remove_item("main", "mcl_core:emerald 1")
		moved = moved + 1
	end
	if moved > 0 then
		meta:set_int("stonks", meta:get_int("stonks") + moved)
	end
	return moved
end

function emeraldbank.get_stonks(pos)
	local meta = core.get_meta(pos)
	local owner = meta:get_string("owner")
	if owner == "" then return end

	inv_emeralds_to_stonks(pos)
	local stonks = meta:get_int("stonks")
	if stonks <= 0 then return end

	local ok, err = emeraldbank.transfer(
		nil,                 -- si es nil usará opts.from_label o SYSTEM (no requiere cuenta ni fondos)
		owner,
		stonks,
		S("Shop earnings"),
		{
			kind = "shops",
			mail = false,      -- no mail
			sound = true,
			from_label = "Shops",
		}
	)

	if not ok then
		core.log("warning", "[emeraldbank] shop payout failed: "..tostring(err)..
			" owner="..tostring(owner).." amount="..tostring(stonks))
		return -- IMPORTANTÍSIMO: NO borres stonks si falló
	end

	meta:set_int("stonks", 0)
	-- if player online
	if core.get_player_by_name(owner) then
		core.chat_send_player(owner, S("You've earned @1 Emeralds with your shops.", stonks))
	end
end
