-- advtrains_info_displays/init.lua


local S = minetest.get_translator("advtrains_info_displays")
local path = minetest.get_modpath("advtrains_info_displays")


advtrains_info_displays = {
	textures = {},  -- Cache
	font = font_api.get_font(minetest.settings:get(
		"advtrains_info_displays_font")),
	color = minetest.colorspec_to_colorstring(minetest.settings:get(
		"advtrains_info_displays_default_color")) or "#FFFF00",
	scroll_speed = tonumber(minetest.settings:get(
		"advtrains_info_displays_scroll_speed")) or 64,
	wait = tonumber(minetest.settings:get(
		"advtrains_info_displays_default_wait")) or 2,
	use_defaults = minetest.settings:get_bool(
		"advtrains_info_displays_use_defaults", true),
}

if minetest.settings:get_bool("advtrains_info_displays_enable_backcolor", true)
then
	advtrains_info_displays.back_color = minetest.colorspec_to_colorstring(
		minetest.settings:get("advtrains_info_displays_default_backcolor"))
		or "#303030"
end

local mod = advtrains_info_displays
local defaults = {}
local max = math.max

function advtrains_info_displays.register_default (name, def)
	if not mod.use_defaults
	or (type(def) ~= "string" and type(def) ~= "function") then
		return
	end
	if defaults[name] then
		return  -- Likely to be changed
	end
	defaults[name] = def
end


local time = 0
local function update_time (dt)
	time = time + dt
end

minetest.register_globalstep(update_time)

function advtrains_info_displays.get_default (slot, wagon, train, tab)
	local def = (tab and tab[slot]) or defaults[slot]
	if def then
		if type(def) == "function" then
			return def(wagon, train)
		else
			return def
		end
	end
end


function advtrains_info_displays.set_textures (textures, wagon, train,
                                               custom_defaults)
	if not mod.font then
		return textures
	end

	local text, mode

	for slot, info in pairs(wagon.info_displays or {}) do
		if not info.ctx then
			info.ctx = tostring(info.width) .. "x" .. tostring(info.height)
			           .. ":" .. (info.color or mod.color)
			           .. ":" .. (info.halign or "center")
		end

		local tslot = train.staticdata
			and train.staticdata.info_displays
			and train.staticdata.info_displays[slot]
		if tslot then
			if tslot.time_current and tslot.time_current > time then
				-- Wagon was reloaded from save, need to correct for time offset
				tslot.time_next = tslot.time_next - tslot.time_current + time
			elseif not tslot.time_current then
				-- Force updating
				tslot.time_next = time
			end
			if time >= tslot.time_next then
				tslot.current = tslot.current + 1
				if not tslot.text[tslot.current] then
					tslot.current = 1
				end
				-- Ensure we don't set current time too far in the past
				tslot.time_current = max(tslot.time_next, time-1)
				tslot.time_next = tslot.time_current + (tslot.times[tslot.current]
					or mod.wait)
			end

			text, mode = tslot.text[tslot.current], tslot.mode[tslot.current]

			if mode then
				if mode == "I" then
					textures[info.slot] = text
				elseif mode == "S" then
					textures[info.slot] = mod.get_texture(info,
						text, (time - tslot.time_current) * mod.scroll_speed)
				end
			else
				textures[info.slot] = mod.get_texture(info, text)
					or textures[info.slot]
			end

		else
			text = mod.get_default(slot, wagon, train, custom_defaults)
			textures[info.slot] = mod.get_texture(info, text)
				or textures[info.slot]
		end

		if mode ~= "I" and mod.back_color then
			if info.bg_fit_text then
				local width = mod.font:get_width(text)
				if width > 0 then
					width = width + 3
					textures[info.slot] = string.format(
						"[combine:%dx%d:0,0=[fill\\:%dx%d\\:0,0\\:%s:2,0=(%s)",
						info.width, info.height, width, info.height,
						info.back_color or mod.back_color,
						textures[info.slot]:gsub(":", "\\:")
					)
				end
			else
				textures[info.slot] = string.format(
					"[fill:%dx%d:0,0:%s^(%s)",
					info.width, info.height, info.back_color or mod.back_color,
					textures[info.slot]
				)
			end
		end
	end
	return textures
end


function advtrains_info_displays.get_texture (info, text, scroll)
	if not text then
		return "blank.png"
	end
	if not mod.textures[info.ctx] then
		mod.textures[info.ctx] = {}
	end

	if scroll then
		if not mod.textures[info.ctx][0] then
			mod.textures[info.ctx][0] = {}
		end
		if not mod.textures[info.ctx][0][text] then
			mod.textures[info.ctx][0][text] =
				mod.font:render(
					text,
					mod.font:get_width(text),
					info.height,
					{
						halign = info.halign,
					}
				):gsub(":", "\\:") .. "^[colorize:" .. (info.color or mod.color)

		end
		return string.format("[combine:%dx%d:%d,0=%s", info.width, info.height,
		        info.width - atfloor(scroll), mod.textures[info.ctx][0][text])
	end

	if not mod.textures[info.ctx][text] then
		mod.textures[info.ctx][text] =
			mod.font:render(
				text,
				info.width,
				info.height,
				{
					color = info.color or mod.color,
					halign = info.halign,
				}
			)
	end
	return mod.textures[info.ctx][text]
end


function advtrains_info_displays.set_info_display (train, slot, text)
	local dat = train.staticdata
	-- Compatibility with old versions of Advtrains
	if not dat then
		train.staticdata = {}
		dat = train.staticdata
	end

	-- In case we need to erase the display, simply remove it
	if not text then
		if dat.info_displays then
			dat.info_displays[slot] = nil
		end
		return
	end

	if not dat.info_displays then
		dat.info_displays = {}
	end
	if not dat.info_displays[slot] then
		dat.info_displays[slot] = {time_next = 0}
	end
	local tslot = dat.info_displays[slot]

	tslot.text = {}
	tslot.times = {}
	tslot.mode = {}

	if type(text) == "string" then
		text = text:split("\n")
	end

	for i, txt in ipairs(text) do
		local parts = txt:split(";", true, 1)
		if parts[2] then
			if parts[1]:sub(1,1) == "I" then
				tslot.mode[i] = "I"
			elseif parts[1]:sub(1,1) == "S" then
				tslot.mode[i] = "S"
			end
			if tslot.mode[i] then
				parts[1] = parts[1]:sub(2)
			end
			tslot.times[i] = tonumber(parts[1])
		end
		tslot.text[i] = tslot.times[i] and parts[2] or txt
	end

	tslot.current = 0
	tslot.time_next = time
end


function advtrains_info_displays.slot_to_table (train, slot)
	local tslot = train.staticdata
		and train.staticdata.info_displays
		and train.staticdata.info_displays[slot]
	if not tslot then
		return {}
	end

	local tab = {}
	for i, text in ipairs(tslot.text) do
		if tslot.times[i] then
			tab[i] = tostring(tslot.times[i]) .. ";" .. text
			if tslot.mode[i] then
				tab[i] = tslot.mode[i] .. tab[i]
			end
		else
			tab[i] = text
		end
	end

	return tab
end



advtrains_info_displays.register_default("line", function (wagon, train)
	return train.line
end)

advtrains_info_displays.register_default("speed", function (wagon, train)
	return train.velocity
end)

advtrains_info_displays.register_default("max_speed", function (wagon, train)
	return train.max_speed
end)


--[[
if advtrains.register_train_field_to_save then
	advtrains.register_train_field_to_save("info_displays")
else
	minetest.log("error",
		"advtrains_info_displays -!- Could not tell advtrains to save info " ..
		"displays: they will not be saved across restarts " ..
		"(installed advtrains version does not appear to support this)")
end
--]]

if atlatc then
	if atlatc.register_function then
		atlatc.register_function("set_info_display",
			function (train, slot, text)
				if advtrains.trains[train] then
					mod.set_info_display(advtrains.trains[train], slot, text)
					return true
				end
				return false
			end
		)
		atlatc.register_function("get_info_display",
			function (train, slot)
				if advtrains.trains[train] then
					return mod.slot_to_table(advtrains.trains[train], slot)
				end
			end
		)
	else
		minetest.log("error",
			"advtrains_info_displays -!- Could not register atlatc functions " ..
			"(installed advtrains version does not appear to support this)")
	end
end

if not mod.font then
	minetest.log("error",
		"advtrains_info_displays -!- Could not obtain any font, please " ..
		"check that you enabled at least one font for Font API.")
end





-- DEBUG TOOL --


local hello = {}


local help_f = "formspec_version[6]size[16,12]textarea[0.5,0.5;15,9.5;;;" ..
	minetest.formspec_escape([[
Info display inspector - Help

This tool is not intended for regular use; if you are running a public server, you should probably consider removing it, by removing everything that comes after line 250 in init.lua.

Trains can have several display slots; each of them is independent from the others, and its contents are shared between all the wagons of the train. Each display slot has a name - e.g, "side", "line"... - by which it is referred. In this help, we will refer to the train you used this tool on as "the train".

A bit of terminology: let us call a "display item" anything that can be displayed on an info display, along with, optionally, the duration it should remain displayed.

Each display slot of a train may display several display items, which are displayed in sequence; for example, assuming a slot has the following three display items:
  * "Subway line 2" (2 seconds)
  * "Chasm of Segfault" (4 seconds)
  * "Express service" (3 seconds)
then it will display "Subway line 2" for 2 seconds, then change to "Chasm of Segfault" for 4 seconds, then to "Express service" for 3 seconds, and then loop back to "Subway line 2".

The list on the left of the formspec shows all the display slots of the train. The rest of the formspec is occupied by a text area which shows all the display items of the selected slot, one per line. You may edit the display items, (if you want to add more, then add more lines,) and then save your changes. Note that changes are saved when you switch to another display slot.

Each display item of the slot is, by default, some text to be displayed on the display slot. The above example (ignoring item duration) would thus be written as follows:
      Subway line 2
      Chasm of segfault
      Express service

In order to specify a duration for a display item, a prefix can be added to its text, which contains the duration of the item (in seconds). The prefix and the text are separated by a ';'. A display item that has no prefix will be displayed for the default duration (3 seconds). For example, the first example can be accurately written as:
      2;Subway line 2
      4;Chasm of segfault
      Express service
the "Express service" item has no duration specified, and will thus be displayed for 3 seconds (the default).

It is also possible to display other things than still text, by appending a letter in front of the display item's prefix, to specify what "other thing" it is. The following letters are currently accepted:
  * 'S' - Scrolling text. The text following the prefix will be scrolled from right to left. The duration this display item is shown is NOT computed from the length of the text or anything else; you MUST specify it. If the text finishes scrolling before the duration has elapsed, then nothing will be displayed until it has elapsed.
  * 'I' - Image. The text following the prefix is actually a Minetest texture string; it will be used "as is", so any valid Minetest texture string will do. Note that, in this case, no background will be added behind the image, as it is the case for plain text.

Some examples:
      S10;Express service to Chasm of Segfault
causes "Express service to Chasm of Segfault" to be scrolled from right to left; after ten seconds have elapsed, it would display the next display item.
      I2;lfs-line-map-U2.png
causes the specified texture (lfs-line-map-U2.png) to be displayed for 2 seconds.

You MUST specify a duration if you specify a letter, or the prefix will not be recognized as such.

If you have any questions or remarks about this help text, please report them in the Minetest forums or on ContentDB.]]) .. "]button[5.5,10.5;3.5,1;back;Back]"


local function get_formspec (pname, train, slot_name)
	local f = "formspec_version[6]size[16,12]"

	local tab = {}
	local temp = {}
	local idx = 0
	local function itab(t)
		tab[#tab+1] = minetest.formspec_escape(string.gsub(t, ",", " "))
	end
	for _, wagon in ipairs(train.trainparts) do
		local _, p = advtrains.get_wagon_prototype(advtrains.wagons[wagon])
		for slot, info in pairs(p.info_displays or {}) do
			if not temp[slot] then
				itab(slot)
				temp[slot] = true
				if slot == slot_name then
					idx = #tab
				end
			end
		end
	end
	local slot_name = slot_name or tab[1]
	if slot_name == tab[1] then idx = 1 end
	f = f .. "textlist[0.5,0.5;3.5,9.5;slots;" .. table.concat(tab, ",")
	f = f .. (slot_name and (";" .. tostring(idx) .. "]") or "]")
	f = f .. "textarea[4.5,0.5;11,9.5;text;;" .. minetest.formspec_escape(table.concat(mod.slot_to_table(train, slot_name), "\n")) .. "]"
	f = f .. "button[0.5,10.5;3.5,1;help;Help]"
	f = f .. "button[7.5,10.5;5,1;sub;Save changes]"
	tab.train = train
	tab.slot = slot_name
	hello[pname] = tab
	return f
end


minetest.register_tool("advtrains_info_displays:temp", {
	description = "Info display inspector",
	inventory_image = "advtrains_copytool.png",
	wield_image = "advtrains_copytool.png",
	stack_max = 1,
	on_use = function(itemstack, user, pointed_thing)
		local name = user:get_player_name()
		if not name or pointed_thing.type ~= "object" then return end
		local ok, missing = minetest.check_player_privs(name, "train_operator")
		if not ok then
			minetest.chat_send_player(name, "You are not allowed to modify the displays of this train. (Missing priviledge: train_operator)")
			return
		end

		local le = pointed_thing.ref:get_luaentity()
		if (le == nil) then
			minetest.chat_send_player(name, "No such lua entity!")
			return
		end

		local wagon = advtrains.wagons[le.id]
		if (not (le.id and advtrains.wagons[le.id])) then
			minetest.chat_send_player(name, string.format("No such wagon: %s", le.id))
			return
		end

		local train = advtrains.trains[wagon.train_id]
		if (not train) then
			minetest.chat_send_player(name, string.format("No such train: %s", wagon.train_id))
			return
		end

		minetest.show_formspec(name, "advtrains_info_displays:player_train_" .. name, get_formspec(name, train))

		return itemstack
	end
})


minetest.register_on_player_receive_fields (
	function (player, formname, fields)
		local name = player:get_player_name()
		if formname ~= "advtrains_info_displays:player_train_" .. name then
			return
		end
		if not minetest.check_player_privs(name, "train_operator") then
			return
		end

		local data = hello[name] or {}
		if fields.quit then
			if fields.text and data.slot then
				mod.set_info_display(data.train, data.slot, fields.text)
			end
			hello[name] = nil
			return true
		end

		if fields.help then
			minetest.show_formspec(
				name, "advtrains_info_displays:player_train_" .. name, help_f)
			return
		end

		local new_slot
		if fields.slots and fields.slots:sub(1, 4) == "CHG:" then
			local new_idx = tonumber(fields.slots:sub(5))
			new_slot = data[new_idx]
		end

		if data.slot and not fields.back then
			mod.set_info_display(data.train, data.slot,
																  fields.text:gsub("%s+$", ""))
		end

		minetest.show_formspec(name, "advtrains_info_displays:player_train_" ..
			name, get_formspec(name, data.train, new_slot or data.slot))

		return true
	end
)
