-- luaatc_textline/api.lua


luaatc_textline.get_info = function (pos)
	local node = core.get_node(pos)
	local ndef = core.registered_nodes[node.name] or {}
	return ndef.luaatc_textline,
	       ndef.luaatc_textline and (ndef.luaatc_textline.lcd_info[node.param2] or {})
end


luaatc_textline.reset_meta = function (pos)
	local meta = core.get_meta(pos)
	local info = luaatc_textline.get_info(pos)
	if info then
		for i = 1, info.displays or 1 do
			meta:set_string('text' .. i, '')
		end
	end
end


luaatc_textline.refresh_size = function (pos, obj, visual_size)
	if not visual_size then
		local _, lcd_info = luaatc_textline.get_info(pos)
		visual_size = lcd_info and lcd_info.visual_size or {x = 1, y = 1, z = 1}
	end
--	local meta = core.get_meta(pos)
--	if meta:get_float('textline_width') ~= 0 then
--		visual_size.x = meta:get_float('textline_width')
--	end
	obj = obj and {obj} or core.get_objects_inside_radius(pos, 0.5)
	for _, o in ipairs(obj) do
		local e = o:get_luaentity()
		if e and e.name == 'luaatc_textline:text' then
			o:set_properties({visual_size = visual_size})
		end
	end
end


luaatc_textline.create_entity = function (pos)
	local info, lcd_info = luaatc_textline.get_info(pos)
	if not lcd_info then
		return
	end
	if not lcd_info.delta then
		lcd_info.delta = {x = 0, y = 0, z = 0}
	end
	local text = core.add_entity({
		x = pos.x + lcd_info.delta.x,
		y = pos.y + lcd_info.delta.y,
		z = pos.z + lcd_info.delta.z
	}, 'luaatc_textline:text')
	luaatc_textline.setup_entity(pos, text, info, lcd_info)
	return text
end


luaatc_textline.setup_entity = function (pos, obj, info, lcd_info)
	if not (info and lcd_info) then
		info, lcd_info = luaatc_textline.get_info(pos)
	end
	if not info then
		return
	end
	obj:set_rotation(lcd_info.rotation or {x = 0, y = 0, z = 0})
	obj:set_properties({glow = info.glow or 0})
	if info.properties then
		obj:set_properties(info.properties)
	end
	obj:set_armor_groups({immortal = 1})
	luaatc_textline.refresh_size(pos, obj, info.visual_size)
end


luaatc_textline.clear_entity = function (pos)
	local objects = core.get_objects_inside_radius(pos, 0.5)
	for _, o in ipairs(objects) do
		local e = o:get_luaentity()
		if e and e.name == 'luaatc_textline:text' then
			o:remove()
		end
	end
end


luaatc_textline.set_text = function (pos, text, ...)
	local info = luaatc_textline.get_info(pos)
	local meta = core.get_meta(pos)
	if not info then
		return
	end
	text = {text, ...}
	if text[1] then
		for i = 1, info.displays or 1 do
			meta:set_string('text' .. i, text[i])
		end
	else
		for i = 1, info.displays or 1 do
			text[i] = meta:contains('text' .. i) and meta:get_string('text' .. i)
		end
		if not text[1] then  -- Compatibility
			text[1] = meta:get_string('text')
			meta:set_string('text', nil)
			meta:set_string('text1', text[1])
		end
	end

	local tex = {}
	if luaatc_textline.use_font_api then
		local font = font_api.get_font(info.font or luaatc_textline.default_font)
		local width = info.width or 96
		local height = info.height or 32
		local params = {
			valign = 'top', halign = 'left',
			color = info.color or luaatc_textline.default_color,
		}
		for i = 1, info.displays or 1 do
			if text[i] then
				tex[i] = font:render(text[i], width, height, params)
			else
				tex[i] = tex[1]
			end
		end
	else
		tex = nil  -- Need fallback
	end
	local found = false
	for _, o in ipairs(core.get_objects_inside_radius(pos, 0.5)) do
		local e = o:get_luaentity()
		if e and e.name and e.name == 'luaatc_textline:text' then
			o:set_properties({textures = tex})
			found = true
		end
	end
	if not found then
		luaatc_textline.create_entity(pos):set_properties({textures = tex})
	end
end


local fire_event = function (pos, evtdata, appr_internal)
	local custom_fct = {
		display = function(text, ...)
			luaatc_textline.set_text(pos, text, ...)
		end,
	}
	atlatc.active.run_in_env(pos, evtdata, custom_fct)
end


luaatc_textline.register_textline = function (name, def)
	if def.groups then
		def.groups.save_in_at_nodedb = def.groups.save_in_at_nodedb or 1
		def.groups.luaatc_textline = def.groups.luaatc_textline or 1
	else
		def.groups = {choppy = 3, dig_immediate = 2, not_blocking_trains = 1,
		              save_in_at_nodedb = 1, luaatc_textline = 1}
	end
	def.after_place_node = def.after_place_node
		or function (pos, placer, itemstack, ...)
			luaatc_textline.create_entity(pos)
			atlatc.active.after_place_node(pos, placer, itemstack, ...)
		end
	def.after_dig_node = def.after_dig_node or atlatc.active.after_dig_node
	def.on_construct = def.on_construct or luaatc_textline.reset_meta
	def.on_destruct = def.on_destruct or luaatc_textline.clear_entity
	def.on_receive_fields = def.on_receive_fields
		or atlatc.active.on_receive_fields
	if def.digiline then
		if not (def.digiline.effector and def.digiline.effector.action) then
			def.digiline.effector = {action = atlatc.active.on_digiline_receive}
		end
	else
		def.digiline = {
			receptor = {},
			effector = {action = atlatc.active.on_digiline_receive},
		}
	end
	if not def.luaautomation then
		def.luaautomation = {}
	end
	if not def.luaautomation.fire_event then
		def.luaautomation.fire_event = fire_event
	end
	core.register_node(name, def)
end


if core.settings:get_bool('luaatc_textline_register_lbm', true) then
	core.register_lbm({
		label = 'Check for LuaATC Textline entities',
		name = 'luaatc_textline:check_entities',
		nodenames = {'group:luaatc_textline'},
		run_at_every_load = core.settings:get_bool(
			'luaatc_textline_lbm_at_every_load', false),
		action = function (pos, node, dtime_s)
			luaatc_textline.set_text(pos)
		end,
	})
end


-- Step timer: tells the entities when to `display` and when to `refresh`.

local step, sld, slr = false, 0, 0
core.register_globalstep(function (dtime)
	sld, slr = sld + dtime, slr + dtime
	if sld > luaatc_textline.refresh_interval then
		step = 'refresh'
		sld = sld - luaatc_textline.refresh_interval
	elseif slr > luaatc_textline.display_interval then
      step = 'display'
		slr = slr - luaatc_textline.display_interval
	else
		step = false
	end
end)


core.register_entity('luaatc_textline:text', {
	initial_properties = {
		collisionbox = {0, 0, 0, 0, 0, 0},
		visual = 'upright_sprite',
		visual_size = {x = 1, y = 1, z = 1},
		textures = {},
	},
	on_activate = function (self)
		local pos = self.object:get_pos()
		luaatc_textline.setup_entity(pos, self.object)
		luaatc_textline.set_text(pos)
	end,
	on_step = function (self)
		if not step then
			return
		end
		local pos = self.object:get_pos():round()
		local n = core.registered_nodes[core.get_node(pos).name]
		if n and n.luaautomation and n.luaautomation.fire_event then
			n.luaautomation.fire_event(pos, {type = step, [step] = true})
		end
	end,
})
