# Dripstone API manual

The dripstone behavior in this mod differs from games like Minecraft by adding
more complexity in a context agnostic manner. This means that in comparison to
a game like Minecraft:

- This dripstone is **computationally very cheap**, as it doesn't care whether
it's part of a stalactite, stalagmite or neither.
- This dripstone grows in **more accurate shapes**, forming more complex shapes
that occur in nature too.
- This dripstone doesn't know whether it is part of a stalactite or stalagmite,
so it **doesn't support falling stalactites**.

![Dripstone demonstration showing various complex shapes](textures/dripstone_example.png)

## Composition

Dripstone has 4 dripstone types:

- Dry dripstone, which is the default type.
- Watered dripstone, which has absorbed water.
- Molten dripstone, which has absorbed lava.
- Hardened dripstone, which has absorbed water & lava at the same time.

Dripstone has 8 different sizes:

- A base dripstone block
- 7  dripstone spike shapes that can grow in height and thickness.

As you can see in the image below, shape sizes from smallest to largest are
spike, tiny, small, medium, great, large, huge, and block.

![Classification of all 32 dripstone nodes](textures/dripstone_classification.png)

From left to right, the dripstone types are dry dripstone, watered dripstone,
molten dripstone and hardened dripstone.

## Dry dripstone

Dry dripstone is the base dripstone type. All dripstone types mimick this type
with a few minor changes.

Dry dripstone blocks are the only type that can absorb liquids. When doing so,
they become watered or molten dripstone blocks.

## Watered & molten dripstone

Dripstone that has absorbed water or lava, can use it for three things. Once it
has done either of the three following options, the dripstone becomes dry.

1. The dripstone can use it to grow 1 size. Only dripstone of size 6 (Large) and
under can do this.
2. The dripstone can pass the liquid down to a dripstone directly below it.
Liquids can never be passed down to hardened dripstones.
3. If the dripstone is a spike size, it can release the droplet downwards to the
ground or to an object (such as a dripstone spike or a cauldron) that can catch
the droplet. If there's no surface within 50 blocks, the droplet evaporates. If
there's a surface that is unable to catch the droplet, a new spike will grow on
top, effectively forming a new stalagmite.

Since dripstone blocks are the only blocks that can absorb liquids, they
function as a sprinkler that grows both the stalactite and stalagmite beneath
it.

**Note:** It is impossible to grow new dripstone blocks using dripstone blocks.
As a result, you must either build your dripstone/lava/water farm in a cave, or
you need to acquire silk touch to be able to break, move and place the dripstone
blocks.

## Hardened dripstone

Similar to how unbreakable a block of obsidian is when lava & water mix,
hardened dripstone is a tough, less breakable type of dripstone that occurs when
a dripstone absorbs both watetr & lava before drying up.

Hardened dripstone cannot grow, it cannot absorb water or pass it on.

## Crafting

Dripstone blocks cannot be crafted.

All other sizes can be crafted using dripstone spikes. You can upgrade a
dripstone item by surrounding it with (the same type of) dripstone spikes.
Huge dripstone cannot be upgraded to a dripstone block, however.

## Lua modders

Everything until this section has been an introduction to how the mod has
implemented dripstone. However, you are welcome to make changes and add your
own types!

Matter of fact, everything until this section has been written using this
publicly available mod. So take a look at [the init.lua file](init.lua) if
you'd prefer to look at a real example.

This mod uses **droplets** to build the dripstone. These are incredibly tiny
drops of liquid that flow down a dripstone. Currently, there are two
types in use: `water` and `lava`.

### dripstone.register_droplet(droplet)

Every droplet type is a unique string. Different droplets interact uniquely
with different dripstones.

```lua
dripstone.register_droplet("magma")
dripstone.register_droplet("mercury")
dripstone.register_droplet("mineral_water")
```

### dripstone.register_source(droplet, nodename)

In order to let droplets stream down dripstone nodes, you need to define which
nodes a full dripstone block can absorb liquid from. Absorbing liquid does not
affect the liquid node: this mod creates a way for players to duplicate liquid
nodes.

Generally, you are recommended to use liquid source blocks to create droplets.
However, you don't need to do this! You could use any node, as long as it has a
name.

```lua
dripstone.register_source("water", "mymod:swamp_water_source")
```

### dripstone.register_catcher(droplet, oldnodename, newnodename)

Similarly, on the other side of the dripstone, you can create a catcher that
can catch any of the liquid drops dripping down. You can use this to create
orchids, or to create other interactions with nodes that might change from a
tiny bit of liquid.

```lua
dripstone.register_catcher("water", "mymod:cauldron_empty", "mymod:water_cauldron")
dripstone.register_catcher("water", "mymod:dirt", "mymod:farmland")
dripstone.register_catcher("lava", "mymod:precious_orchid", "mymod:dead_bush")
```

### dripstone.register_dripstone(flavor, def)

You can define your own dripstone type. You need to pick a flavor (which you
shouldn't namespace) and you need to offer a dripstone definition, which is
defined as follows:

```
{
    -- What item is dropped when the dripstone is broken.
    -- When left nil, the spike of the dripstone type is dropped.
    drop = "dry"

    -- What flavor to become when using liquid to grow.
    -- Leave to nil when unable to grow.
    grow_to = "dry"

    -- When receiving a droplet of a given type, transform into a different
    -- dripstone type. When a droplet is unspecified, the block cannot
    -- receive the droplet.
    on_droplet_receive = {
        water = "watered",
        lava = "molten",
    }

    -- Sounds that the dripstone makes
    sounds = <standard sound definition for a node>

    -- Node tiles for layout
    tiles = <node tile layout>

    -- Droplet type that the dripstone flavor can pass down.
    -- When the droplet is passed down, the dripstone converts to the
    -- "grow_to" type
    trickle_down = "water"

    -- Speed of how often a droplet trickles down.
    trickle_speed = 5
}
```

For examples, refer to [this mod's init.lua file](init.lua).

### dripstone.size_to_name(flavor, size)

Lastly, this function serves as a tool to help you find the node name of a
dripstone of a given flavor and size. Remember that dripstone size range from 1
to 8.

```lua
dripstone.size_to_name("dry", 1)
-- Returns "dripstone:dry_dripstone_spike"

dripstone.size_to_name("lol", 3)
-- Returns "dripstone:small_lol_dripstone"

dripstone.size_to_name("cool", 8)
-- Returns "dripstone:cool_dripstone_block"
```

Note how the function even returns a string if the dripstone type hasn't been
registered.
