-- xpanes/init.lua

-- Load support for MT game translation.
local S = minetest.get_translator("xpanes")


local function is_pane(pos)
	return minetest.get_item_group(minetest.get_node(pos).name, "pane") > 0
end

local function connects_dir(pos, name, dir)
	local aside = vector.add(pos, minetest.facedir_to_dir(dir))
	if is_pane(aside) then
		return true
	end

	local connects_to = minetest.registered_nodes[name].connects_to
	if not connects_to then
		return false
	end
	local list = minetest.find_nodes_in_area(aside, aside, connects_to)

	if #list > 0 then
		return true
	end

	return false
end

local function swap(pos, node, name, param2)
	if node.name == name and node.param2 == param2 then
		return
	end

	minetest.swap_node(pos, {name = name, param2 = param2})
end

local function update_pane(pos)
	if not is_pane(pos) then
		return
	end
	local node = minetest.get_node(pos)
	local name = node.name
	if name:sub(-5) == "_flat" then
		name = name:sub(1, -6)
	end

	local any = node.param2
	local c = {}
	local count = 0
	for dir = 0, 3 do
		c[dir] = connects_dir(pos, name, dir)
		if c[dir] then
			any = dir
			count = count + 1
		end
	end

	if count == 0 then
		swap(pos, node, name .. "_flat", any)
	elseif count == 1 then
		swap(pos, node, name .. "_flat", (any + 1) % 4)
	elseif count == 2 then
		if (c[0] and c[2]) or (c[1] and c[3]) then
			swap(pos, node, name .. "_flat", (any + 1) % 4)
		else
			swap(pos, node, name, 0)
		end
	else
		swap(pos, node, name, 0)
	end
end

minetest.register_on_placenode(function(pos, node)
	if minetest.get_item_group(node, "pane") then
		update_pane(pos)
	end
	for i = 0, 3 do
		local dir = minetest.facedir_to_dir(i)
		update_pane(vector.add(pos, dir))
	end
end)

minetest.register_on_dignode(function(pos)
	for i = 0, 3 do
		local dir = minetest.facedir_to_dir(i)
		update_pane(vector.add(pos, dir))
	end
end)

xpanes = {}
function xpanes.register_pane(name, def)
	for i = 1, 15 do
		minetest.register_alias("hades_xpanes:" .. name .. "_" .. i, "hades_xpanes:" .. name .. "_flat")
	end

	local flatgroups = table.copy(def.groups)
	flatgroups.pane = 1
	minetest.register_node(":hades_xpanes:" .. name .. "_flat", {
		description = def.description,
		drawtype = "nodebox",
		paramtype = "light",
		is_ground_content = false,
		sunlight_propagates = true,
		inventory_image = def.inventory_image,
		wield_image = def.wield_image,
		paramtype2 = "facedir",
		tiles = {
			def.textures[3],
			def.textures[3],
			def.textures[3],
			def.textures[3],
			def.textures[1],
			def.textures[1]
		},
		groups = flatgroups,
		drop = "hades_xpanes:" .. name .. "_flat",
		sounds = def.sounds,
		use_texture_alpha = def.use_texture_alpha and "blend" or "clip",
		node_box = {
			type = "fixed",
			fixed = {{-1/2, -1/2, -1/32, 1/2, 1/2, 1/32}},
		},
		selection_box = {
			type = "fixed",
			fixed = {{-1/2, -1/2, -1/32, 1/2, 1/2, 1/32}},
		},
		connect_sides = { "left", "right" },
	})

	local groups = table.copy(def.groups)
	groups.pane = 1
	groups.not_in_creative_inventory = 1
	minetest.register_node(":hades_xpanes:" .. name, {
		drawtype = "nodebox",
		paramtype = "light",
		is_ground_content = false,
		sunlight_propagates = true,
		description = def.description,
		tiles = {
			def.textures[3],
			def.textures[3],
			def.textures[1]
		},
		groups = groups,
		drop = "hades_xpanes:" .. name .. "_flat",
		sounds = def.sounds,
		use_texture_alpha = def.use_texture_alpha and "blend" or "clip",
		node_box = {
			type = "connected",
			fixed = {{-1/32, -1/2, -1/32, 1/32, 1/2, 1/32}},
			connect_front = {{-1/32, -1/2, -1/2, 1/32, 1/2, -1/32}},
			connect_left = {{-1/2, -1/2, -1/32, -1/32, 1/2, 1/32}},
			connect_back = {{-1/32, -1/2, 1/32, 1/32, 1/2, 1/2}},
			connect_right = {{1/32, -1/2, -1/32, 1/2, 1/2, 1/32}},
		},
		connects_to = {"group:pane", "group:stone", "group:glass", "group:wood", "group:tree"},
	})

	minetest.register_craft({
		output = "hades_xpanes:" .. name .. "_flat 16",
		recipe = def.recipe
	})
end

xpanes.register_pane("pane", {
	description = S("Glass Pane"),
	textures = {"default_glass.png", "", "xpanes_edge.png"},
	inventory_image = "default_glass.png",
	wield_image = "default_glass.png",
	sounds = hades_sounds.node_sound_glass_defaults(),
	groups = {snappy=2, cracky=3, oddly_breakable_by_hand=3},
	recipe = {
		{"hades_core:glass", "hades_core:glass", "hades_core:glass"},
		{"hades_core:glass", "hades_core:glass", "hades_core:glass"}
	}
})

xpanes.register_pane("obsidian_pane", {
	description = S("Obsidian Glass Pane"),
	textures = {"default_obsidian_glass.png", "", "xpanes_edge_obsidian.png"},
	inventory_image = "default_obsidian_glass.png",
	wield_image = "default_obsidian_glass.png",
	sounds = hades_sounds.node_sound_glass_defaults(),
	groups = {snappy=2, cracky=3},
	recipe = {
		{"hades_core:obsidian_glass", "hades_core:obsidian_glass", "hades_core:obsidian_glass"},
		{"hades_core:obsidian_glass", "hades_core:obsidian_glass", "hades_core:obsidian_glass"}
	}
})

xpanes.register_pane("bar", {
	description = S("Steel Bars"),
	textures = {"xpanes_bar.png", "", "xpanes_bar_top.png"},
	inventory_image = "xpanes_bar.png",
	wield_image = "xpanes_bar.png",
	groups = {cracky=2},
	sounds = hades_sounds.node_sound_metal_defaults(),
	recipe = {
		{"hades_core:steel_ingot", "hades_core:steel_ingot", "hades_core:steel_ingot"},
		{"hades_core:steel_ingot", "hades_core:steel_ingot", "hades_core:steel_ingot"}
	}
})

minetest.register_lbm({
	name = "hades_xpanes:gen2",
	nodenames = {"group:pane"},
	action = function(pos, node)
		update_pane(pos)
		for i = 0, 3 do
			local dir = minetest.facedir_to_dir(i)
			update_pane(vector.add(pos, dir))
		end
	end
})

-- Register steel bar doors and trapdoors

if minetest.get_modpath("hades_doors") then

	hades_doors.register("hades_xpanes:door_steel_bar", {
		tiles = {{name = "xpanes_door_steel_bar.png", backface_culling = true}},
		use_texture_alpha = "clip",
		description = S("Steel Bar Door"),
		inventory_image = "xpanes_item_steel_bar.png",
		protected = true,
		groups = {node = 1, cracky = 1, level = 2},
		sounds = hades_sounds.node_sound_metal_defaults(),
		sound_open = "xpanes_steel_bar_door_open",
		sound_close = "xpanes_steel_bar_door_close",
		recipe = {
			{"hades_xpanes:bar_flat", "hades_xpanes:bar_flat"},
			{"hades_xpanes:bar_flat", "hades_xpanes:bar_flat"},
			{"hades_xpanes:bar_flat", "hades_xpanes:bar_flat"},
		},
	})

	hades_doors.register_trapdoor("hades_xpanes:trapdoor_steel_bar", {
		description = S("Steel Bar Trapdoor"),
		inventory_image = "xpanes_trapdoor_steel_bar.png",
		wield_image = "xpanes_trapdoor_steel_bar.png",
		tile_front = "xpanes_trapdoor_steel_bar.png",
		tile_side = "xpanes_trapdoor_steel_bar_side.png",
		protected = true,
		groups = {node = 1, cracky = 1, level = 2, door = 1},
		sounds = hades_sounds.node_sound_metal_defaults(),
		sound_open = "xpanes_steel_bar_door_open",
		sound_close = "xpanes_steel_bar_door_close",
	})

	minetest.register_craft({
		output = "hades_xpanes:trapdoor_steel_bar",
		recipe = {
			{"hades_xpanes:bar_flat", "hades_xpanes:bar_flat"},
			{"hades_xpanes:bar_flat", "hades_xpanes:bar_flat"},
		}
	})
end
