-- Debug tool for visualizing hitboxes on entities/players
-- Admin tool - no recipe, use /give command

local function print_hits(hits)
	if hits and #hits > 0 then
		print(string.format("  Total hits: %d", #hits))
		print("----------------------------------------")
		
		for i, hit in ipairs(hits) do
			print(string.format("  Hit #%d: '%s'", i, hit.name))
			print(string.format("    Distance: %.2f", hit.distance))
			print(string.format("    Priority: %d", hit.orig.priority or 0))
			print(string.format("    Hit axis: %s", hit.hit_axis or "unknown"))
			print(string.format("    Hit position (normalized): x=%.3f, y=%.3f, z=%.3f", 
				hit.hit_relative.x, hit.hit_relative.y, hit.hit_relative.z))
		end
	else
		print("  No hits detected")
	end
end

core.register_tool("hitboxes_lib:hitbox_visualizer_tool", {
	description = "Hitbox Visualizer Tool\n" ..
	              "Left-click: Show hitboxes visually (attached, 30s)\n" ..
	              "Right-click: Test all hit detection modes and print results\n" ..
	              "  (raycast, box, and sphere)\n" ..
	              "Shift+Left-click: Slow down target",
	inventory_image = "hitboxes_lib_hitbox_tool.png",
	range = 10.0,
	
	on_secondary_use = function(itemstack, user, pointed_thing)
		if pointed_thing.type ~= "object" then
			return
		end
		
		local target = pointed_thing.ref
		if not target or not target:is_player() and not target:get_luaentity() then
			return
		end
		
		local player_name = user:get_player_name()
		
		-- Determine hitbox group name
		local hitgroup_name = hitboxes_lib.get_hitgroup_name(target)
		if not hitgroup_name then
			core.chat_send_player(player_name, "[Hitbox Visualizer] Invalid entity for hitbox detection")
			print("[Hitbox Visualizer] Invalid entity for hitbox detection")
			return
		end
		
		-- Check if hitboxes are registered
		if not hitboxes_lib.hitbox_groups[hitgroup_name] then
			core.chat_send_player(player_name, string.format("[Hitbox Visualizer] No hitboxes found for '%s'", hitgroup_name))
			print(string.format("[Hitbox Visualizer] No hitboxes registered for group '%s'", hitgroup_name))
			return
		end
		
		-- Calculate positions and prepare hit_data structure
		local entity_pos = target:get_pos()
		local puncher_pos = user:get_pos()
		local eye_height = user:get_properties().eye_height
		local ref_pos = puncher_pos
		local hit_from_pos = vector.add(puncher_pos, vector.new(0, eye_height, 0))
		local hit_from_dir = user:get_look_dir()
		
		-- Get target rotation
		local target_rot
		if target:is_player() then
			-- For players, use yaw only
			local look = target:get_look_horizontal()
			target_rot = {x=0, y=look, z=0}
		else
			target_rot = target:get_rotation()
		end
		
		-- Calculate relative position for hitboxes
		local move = vector.subtract(entity_pos, ref_pos)
		
		-- Get transformed hitboxes at relative position
		local hitboxes = hitboxes_lib.get_transformed_boxes(hitgroup_name, move, target_rot)
		
		-- Prepare hit_data with new structure
		local hit_data = {
			ref_pos = ref_pos,
			hit_from_pos = hit_from_pos,
			hit_from_dir = hit_from_dir,
		}
		
		-- Perform raycast
		hit_data.range = 10
		local hits = hitboxes_lib.raycast_hit(hitboxes, hit_data, 10)
		
		-- Prepare hit_data for box test (box in local coordinates)
		local _box_hit_local = hitboxes_lib.collisionbox_to_box({-0.1, -0.1, -0.1, 0.1, 0.1, 0.1})
		local hit_data_box = {
			ref_pos = ref_pos,
			hit_from_pos = hit_from_pos,
			hit_from_dir = hit_from_dir,
			box = _box_hit_local,
			box_rot = user:get_rotation(),
			range = 10,
		}
		local box_hits = hitboxes_lib.box_hit(hitboxes, hit_data_box)
		
		-- Perform sphere hit test (radius 0.1)
		local hit_data_sphere = {
			ref_pos = ref_pos,
			hit_from_pos = hit_from_pos,
			hit_from_dir = hit_from_dir,
			sphere_radius = 0.1,
			range = 10,
		}

		local sphere_hits = hitboxes_lib.sphere_hit(hitboxes, hit_data_sphere)

		-- Print hit information to terminal
		print("========================================")
		print(string.format("[Hitbox Visualizer] Hit detection for '%s'", hitgroup_name))
		print(string.format("  Entity position: (%.2f, %.2f, %.2f)", entity_pos.x, entity_pos.y, entity_pos.z))
		print(string.format("  Reference position: (%.2f, %.2f, %.2f)", ref_pos.x, ref_pos.y, ref_pos.z))
		print(string.format("  Hit from position: (%.2f, %.2f, %.2f)", hit_from_pos.x, hit_from_pos.y, hit_from_pos.z))
		print(string.format("  Hit from direction: (%.2f, %.2f, %.2f)", hit_from_dir.x, hit_from_dir.y, hit_from_dir.z))
		print(string.format("  Relative move: (%.2f, %.2f, %.2f)", move.x, move.y, move.z))
		
		print("========================================")
		print("Raycast Hits:")
		print_hits(hits)
		print("========================================")
		print("Box Hits:")
		print_hits(box_hits)
		print("========================================")
		print("Sphere Hits:")
		print_hits(sphere_hits)
		
		print("========================================")
	end,
	
	on_use = function(itemstack, user, pointed_thing)
		if pointed_thing.type ~= "object" then
			return
		end
		
		local target = pointed_thing.ref
		if not target or not target:is_player() and not target:get_luaentity() then
			return
		end
		
	-- Check if shift is pressed (slow down entity)
	local player_name = user:get_player_name()
	local player_ctrl = user:get_player_control()
	
	if player_ctrl.sneak then
		-- Slow down the target entity
		if target:is_player() then
			-- Slow down player
			target:set_physics_override({speed = 0.1})
			core.chat_send_player(player_name, "[Hitbox Visualizer] Slowed down player")
		else
			local entity = target:get_luaentity()
			if entity then
				-- Try to slow down entity by modifying walk_speed and run_speed
				local props = target:get_properties()
				local modified = false
				
				if props.stepheight then -- Check if it's a mob-like entity
					-- Store original speeds if not already stored
					if not entity._original_walk_speed then
						entity._original_walk_speed = entity.walk_velocity or 1
						entity._original_run_speed = entity.run_velocity or 2
					end
					
					-- Set very low speeds
					entity.walk_velocity = 0.1
					entity.run_velocity = 0.1
					modified = true
				end
				
				if modified then
					core.chat_send_player(player_name, "[Hitbox Visualizer] Slowed down entity")
				else
					core.chat_send_player(player_name, "[Hitbox Visualizer] Cannot slow down this entity type")
				end
			end
		end
	end	-- Determine hitbox group name
	local hitgroup_name = hitboxes_lib.get_hitgroup_name(target)
	if not hitgroup_name then
		core.chat_send_player(player_name, "[Hitbox Visualizer] Invalid entity for hitbox detection")
		return
	end
	
	-- Check if hitboxes are registered
	if not hitboxes_lib.hitbox_groups[hitgroup_name] then
		core.chat_send_player(player_name, string.format("[Hitbox Visualizer] No hitboxes registered for group '%s'", hitgroup_name))
		return
	end
	
	-- Visualize hitboxes (attached mode, 30 seconds)
	hitboxes_lib.visualize_object_hitboxes(hitgroup_name, target, 30, true)
	
	core.chat_send_player(player_name, string.format("[Hitbox Visualizer] Showing hitbox(es) for '%s' (30s, attached)", hitgroup_name))
	end,
})
