
local S = manage_entities.translator
local settings = manage_entities.settings

-- Chat command to check entities on the server
core.register_chatcommand("manage_entities_check", {
	params = "[<check_radius>] [<entities_limit>]",
	description = S("Check all active entities and report positions where entity count exceeds limit"),
	privs = {server=true},
	func = function(player_name, params)
		local args = params:split(" ")
		
		-- Parse optional parameters or use defaults
		local check_radius = tonumber(args[1]) or settings.check_radius
		local entities_limit = tonumber(args[2]) or settings.entities_limit
		
		if check_radius <= 0 or entities_limit <= 0 then
			return false, S("Radius and limit must be greater than 0")
		end
		
		local checked_entities = {}
		local reported_areas = {}  -- Track which entities we've already reported as part of a problematic area
		local problem_areas = {}   -- Store info about problem areas
		local total_checks = 0
		
		core.chat_send_player(player_name, S("Starting entity check with radius=@1, limit=@2", 
			check_radius, entities_limit))
		
		-- Check each entity
		for guid, obj in pairs(core.objects_by_guid) do
			if obj and obj:get_pos() then
				local pos = obj:get_pos()
				local entity = obj:get_luaentity()
				
				if entity and entity.name then
					total_checks = total_checks + 1
					
					-- Track entity types
					if not checked_entities[entity.name] then
						checked_entities[entity.name] = 0
					end
					checked_entities[entity.name] = checked_entities[entity.name] + 1
					
					-- Skip if this entity was already reported as part of another problematic area
					if not reported_areas[guid] then
						-- Count entities in radius around this entity
						local nearby_objects = core.get_objects_inside_radius(pos, check_radius)
						local nearby_count = #nearby_objects
						
						-- Check if limit is exceeded
						if nearby_count > entities_limit then
							-- Mark all entities in this area as reported to avoid duplicate reports
							for _, nearby_obj in ipairs(nearby_objects) do
								if nearby_obj then
									local nearby_guid = nearby_obj:get_guid()
									if nearby_guid then
										reported_areas[nearby_guid] = true
									end
								end
							end
							
							-- Store problem area info
							local pos_str = core.pos_to_string(vector.round(pos))
							table.insert(problem_areas, {
								pos = pos_str,
								count = nearby_count,
								entity_type = entity.name
							})
						end
					end
				end
			end
		end
		
		-- Report all problem areas
		if #problem_areas > 0 then
			for _, area in ipairs(problem_areas) do
				core.chat_send_player(player_name, 
					S("Position @1: @2 entities (entity: @3)", 
						area.pos, area.count, area.entity_type))
			end
		end
		
		-- Send summary
		core.chat_send_player(player_name, S("--- Check Summary ---"))
		core.chat_send_player(player_name, S("Total entities checked: @1", total_checks))
		
		-- Count entities in problematic areas
		local entities_in_problem_areas = 0
		for _ in pairs(reported_areas) do
			entities_in_problem_areas = entities_in_problem_areas + 1
		end
		core.chat_send_player(player_name, S("Entities exceeding limit: @1", entities_in_problem_areas))
		core.chat_send_player(player_name, S("Unique positions reported: @1", #problem_areas))
		
		-- List entity types found
		core.chat_send_player(player_name, S("--- Entity Types Found ---"))
		for entity_name, count in pairs(checked_entities) do
			core.chat_send_player(player_name, string.format("  %s: %d", entity_name, count))
		end
		
		return true, S("Entity check completed")
	end,
})
