
local S = manage_entities.translator
local settings = manage_entities.settings

-- Chat command to check entities on the server
core.register_chatcommand("manage_entities_check", {
	params = "[<check_radius>] [<entities_limit>]",
	description = S("Check all active entities and report positions where entity count exceeds limit"),
	privs = {server=true},
	func = function(player_name, params)
		local args = params:split(" ")
		
		-- Parse optional parameters or use defaults
		local check_radius = tonumber(args[1]) or settings.check_radius
		local entities_limit = tonumber(args[2]) or settings.entities_limit
		
		if check_radius <= 0 or entities_limit <= 0 then
			return false, S("Radius and limit must be greater than 0")
		end
		
		local checked_entities = {}
		local reported_areas = {}  -- Track which entities we've already reported as part of a problematic area
		local problem_areas = {}   -- Store info about problem areas
		local total_checks = 0
		
		core.chat_send_player(player_name, S("Starting entity check with radius=@1, limit=@2", 
			check_radius, entities_limit))
		
		-- Check each entity
		for guid, obj in pairs(core.objects_by_guid) do
			if obj and obj:get_pos() then
				local pos = obj:get_pos()
				local entity = obj:get_luaentity()
				
				if entity and entity.name then
					total_checks = total_checks + 1
					
					-- Track entity types
					if not checked_entities[entity.name] then
						checked_entities[entity.name] = 0
					end
					checked_entities[entity.name] = checked_entities[entity.name] + 1
					
					-- Skip if this entity was already reported as part of another problematic area
					if not reported_areas[guid] then
						-- Count entities in radius around this entity
						local nearby_objects = core.get_objects_inside_radius(pos, check_radius)
						local nearby_count = #nearby_objects
						
						-- Check if limit is exceeded
						if nearby_count > entities_limit then
							-- Mark all entities in this area as reported to avoid duplicate reports
							for _, nearby_obj in ipairs(nearby_objects) do
								if nearby_obj then
									local nearby_guid = nearby_obj:get_guid()
									if nearby_guid then
										reported_areas[nearby_guid] = true
									end
								end
							end
							
							-- Store problem area info
							local pos_str = core.pos_to_string(vector.round(pos))
							table.insert(problem_areas, {
								pos = pos_str,
								count = nearby_count,
								entity_type = entity.name
							})
						end
					end
				end
			end
		end
		
		-- Report all problem areas
		if #problem_areas > 0 then
			for _, area in ipairs(problem_areas) do
				core.chat_send_player(player_name, 
					S("Position @1: @2 entities (entity: @3)", 
						area.pos, area.count, area.entity_type))
			end
		end
		
		-- Send summary
		core.chat_send_player(player_name, S("--- Check Summary ---"))
		core.chat_send_player(player_name, S("Total entities checked: @1", total_checks))
		
		-- Count entities in problematic areas
		local entities_in_problem_areas = 0
		for _ in pairs(reported_areas) do
			entities_in_problem_areas = entities_in_problem_areas + 1
		end
		core.chat_send_player(player_name, S("Entities exceeding limit: @1", entities_in_problem_areas))
		core.chat_send_player(player_name, S("Unique positions reported: @1", #problem_areas))
		
		-- List entity types found
		core.chat_send_player(player_name, S("--- Entity Types Found ---"))
		for entity_name, count in pairs(checked_entities) do
			core.chat_send_player(player_name, string.format("  %s: %d", entity_name, count))
		end
		
		return true, S("Entity check completed")
	end,
})

if manage_entities.settings.config_command then
	-- Chat command to modify runtime limits and radii
	core.register_chatcommand("manage_entities_config", {
		params = "<set|get|list> [entity_name] [limit=<value>] [radius=<value>]",
		description = S("Configure entity limits and check radius at runtime"),
		privs = {server=true},
		func = function(player_name, params)
			local args = params:split(" ")
			
			if #args < 1 then
				return false, S("Usage: /manage_entities_config <set|get|list> [entity_name] [limit=<value>] [radius=<value>]")
			end
			
			local action = args[1]
			
			-- List all current configurations
			if action == "list" then
				core.chat_send_player(player_name, S("--- Per-Entity Settings ---"))
				for entity_name, config in pairs(settings.entity_configs) do
					core.chat_send_player(player_name, string.format("  %s: limit=%d, radius=%.2f",
						entity_name, config.entities_limit, config.check_radius))
				end
				return true
			end
			
			-- Get specific entity configuration
			if action == "get" then
				if #args < 2 then
					return false, S("Usage: /manage_entities_config get <entity_name>")
				end
				
				local entity_name = args[2]
				
				local config = settings.entity_configs[entity_name]
				if config then
					return true, S("@1: limit=@2, radius=@3", 
						entity_name, config.entities_limit, config.check_radius)
				else
					return false, S("Entity '@1' not found in configuration", entity_name)
				end
			end
			
			-- Set entity configuration
			if action == "set" then
				if #args < 3 then
					return false, S("Usage: /manage_entities_config set <entity_name> limit=<value> [radius=<value>]")
				end
				
				local entity_name = args[2]
				local new_limit = nil
				local new_radius = nil
				
				-- Parse limit and radius from remaining arguments
				for i = 3, #args do
					local key, value = args[i]:match("^([^=]+)=(.+)$")
					if key and value then
						if key == "limit" then
							new_limit = tonumber(value)
						elseif key == "radius" then
							new_radius = tonumber(value)
						end
					end
				end
				
				if not new_limit and not new_radius then
					return false, S("At least one of limit=<value> or radius=<value> must be specified")
				end
				
				-- Update per-entity settings
				if not settings.entity_configs[entity_name] then
					settings.entity_configs[entity_name] = {
						entities_limit = settings.entities_limit,
						check_radius = settings.check_radius
					}
				end
				
				if new_limit then
					if new_limit <= 0 then
						return false, S("Limit must be greater than 0")
					end
					settings.entity_configs[entity_name].entities_limit = new_limit
				end
				if new_radius then
					if new_radius <= 0 then
						return false, S("Radius must be greater than 0")
					end
					settings.entity_configs[entity_name].check_radius = new_radius
				end
				
				return true, S("Settings updated for '@1': limit=@2, radius=@3", 
					entity_name, 
					settings.entity_configs[entity_name].entities_limit,
					settings.entity_configs[entity_name].check_radius)
			end
			
			return false, S("Unknown action '@1'. Use: set, get, or list", action)
		end,
	})
end
