local channels = {}

local colors = {}
function mt_channels.get_color_by_colorname(colorname)
    return colors[colorname]
end

function mt_channels.register_color(color_name, color_code)
    if type(color_code) == "table" then
        color_code = string.format("#%02X%02X%02X", color_code[1], color_code[2], color_code[3])
    end
    colors[color_name] = color_code
end

function mt_channels.get_custom_colors()
    if mt_channels.colors == "basic" then
        return nil
    end
    local color_list = {}
    for color_name, _ in pairs(colors) do
        table.insert(color_list, color_name)
    end
    table.sort(color_list)
    return table.concat(color_list, ",")
end

function mt_channels.get_color(channel_name)
    if mt_channels.exists(channel_name) then
        return channels[channel_name].color
    end
    return false
end

function mt_channels.create(channel_name, owner, color)
    if mt_channels.colors == "custom" then
        local custom_color = mt_channels.get_color_by_colorname(color)
        if custom_color then
            color = custom_color
        end
    end
    channels[channel_name] = {owner=owner, color=color, players={owner}}
end

function mt_channels.delete(channel_name)
    for channel in pairs(channels[channel_name]) do
        channels[channel_name][channel] = nil
    end
    channels[channel_name] = nil
end

function mt_channels.edit(channel_name, owner, color)
    local old_channel_name = mt_channels.get_channel_by_owner(owner)
    if not old_channel_name then
        mt_channels.create(channel_name, owner, color)
    end
    local old_players = {}
    if channels[old_channel_name] and channels[old_channel_name].players then
        old_players = channels[old_channel_name].players
    end
    if channel_name == old_channel_name then
        channels[channel_name].color = color
    else
        channels[channel_name] = {owner=owner, color=color, players=old_players}
        mt_channels.delete(old_channel_name)
    end
end

function mt_channels.exists(channel_name)
    return channels[channel_name] ~= nil
end

function mt_channels.is_owner(player_name, channel_name)
    if channels[channel_name].owner == player_name then
        return true
    end
    return false
end

function mt_channels.add_player(player_name, channel_name)
    table.insert(channels[channel_name].players, player_name)
end

function mt_channels.remove_player(player_name, channel_name)
    for i, player in pairs(channels[channel_name].players) do
        if player == player_name then
            table.remove(channels[channel_name].players, i)
        end
    end
    return nil
end

function mt_channels.get_players(channel_name)
    if channels[channel_name] and channels[channel_name].players then
        return channels[channel_name].players
    end
    return {}
end

function mt_channels.get_channel_by_owner(owner)
    for channel_name, channel in pairs(channels) do
        if channel.owner == owner then
            return channel_name
        end
    end
    return false
end

function mt_channels.get_owner(channel_name)
    if mt_channels.exists(channel_name) then
        return channels[channel_name].owner
    end
    return false
end

function mt_channels.player_in_channel(player_name, channel_name)
    for _, player in pairs(channels[channel_name].players) do
        if player == player_name then
            return true
        end
    end
    return false
end

function mt_channels.get_player_channel(player_name)
    for channel_name, channel in pairs(channels) do
        for _, player in pairs(channel.players) do
            if player == player_name then
                return channel_name
            end
        end
    end
    return nil
end

function mt_channels.send_message(channel_name, author, message, type)
    local color = mt_channels.get_color(channel_name)
    for _, player_name in pairs(channels[channel_name].players) do
        if type == "info" then
            core.chat_send_player(player_name, core.colorize(color, "(" .. channel_name .. ") " .. message))
        else
            core.chat_send_player(player_name, core.colorize(color, "(" .. channel_name .. ") " .. "<" .. author .. "> " .. message))
        end
    end
    if type == "info" then
        core.log("action", "(" .. channel_name .. ") " .. message)
        return
    end
    core.log("action", "(" .. channel_name .. ") " .. "<" .. author .. "> " .. message)
end

local invite_pending = {}

function mt_channels.send_invite(channel_name, inviter, invitee)
    invite_pending[invitee] = {channel=channel_name, inviter=inviter}
end

function mt_channels.accept_invite(name)
    local channel_name = invite_pending[name].channel
    local inviter = invite_pending[name].inviter
    mt_channels.add_player(name, channel_name)
    core.chat_send_player(name, "You accept the invitation to join the channel " .. channel_name .. ".")
    core.chat_send_player(inviter, name .. " accepts your invitation to join the channel.")
    mt_channels.send_message(channel_name, name, name .. " joined the channel.", "info")
    invite_pending[name] = nil
end

function mt_channels.decline_invite(name)
    local channel_name = invite_pending[name].channel
    local inviter = invite_pending[name].inviter
    core.chat_send_player(name, "You decline the invitation to join the channel " .. channel_name .. ".")
    core.chat_send_player(inviter, name .. " declines your invitation to join the channel.")
    invite_pending[name] = nil
end

function mt_channels.get_invite_pending(name)
    return invite_pending[name]
end

function mt_channels.get_all_channels()
    return channels
end

function mt_channels.get_channel(channel_name)
    return channels[channel_name]
end