--[[
    Horn Track for Advanced Trains

    Copyright © 2024, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Public API functions.
-- @module api


local S = advtrains_horntrack.S;
local util = advtrains_horntrack.util;


advtrains_horntrack.DEFAULT_PATTERN = '1';
advtrains_horntrack.IMPLICIT_GAP = 0.5;
advtrains_horntrack.FADE_RATE = 4.0;
advtrains_horntrack.TEST_SOUND =
{
    name           = 'advtrains_horntrack_test';
    gain           = 0.5;
    description    = S'Testing Horn';
    merge_subtitle = true;
};


local g_stop_funcs = {};


--- Activates a train's horn.
-- @param pattern [HornPattern] The horn pattern to play.
-- @param train   [Train]       The train whose horn to activate.
-- @return        [boolean]     true on success.
function advtrains_horntrack.toot(pattern, train)
    local horn_sound, wagon = advtrains_horntrack.get_train_horn(train);
    if not horn_sound then
        return false;
    end;

    advtrains_horntrack.toot_at(pattern, horn_sound, wagon);
    return true;
end;


--- Tests a horn pattern by playing it without a train.
-- @param pattern [HornPattern] The horn pattern to play.
-- @param pos     [vector]      Where to play the sound.
function advtrains_horntrack.toot_test(pattern, pos)
    advtrains_horntrack.toot_at(pattern, advtrains_horntrack.TEST_SOUND, pos);
end;


--- Plays a horn sound at a position or attached to an object.
-- @param pattern      [HornPattern]     The horn pattern to play.
-- @param horn_sound   [SimpleSoundSpec] The horn sound to play.
-- @param wagon_or_pos [Wagon|vector]    The wagon to attach the sound to, or the absolute location to play at.
function advtrains_horntrack.toot_at(pattern, horn_sound, wagon_or_pos)
    local pos;
    local wagon;
    if vector.check(wagon_or_pos) then
        pos = wagon_or_pos;
    else
        wagon = wagon_or_pos;
    end;

    local index = 1;
    local handle;
    local gain = 0;
    local timer;
    local object = wagon and advtrains.wagon_objects[wagon.id];
    local entity = object and object:get_luaentity();
    local horn_id = (wagon and wagon.id) or (pos and minetest.pos_to_string(pos)) or '';

    local function _stop_sound(stop_handle, stop_gain)
        if stop_handle then
            minetest.sound_fade(stop_handle, advtrains_horntrack.FADE_RATE, 0);
            minetest.after(gain / advtrains_horntrack.FADE_RATE + 0.0625, function() minetest.sound_stop(stop_handle); end);
        end;
        handle = nil;
        gain = 0;
    end;

    local function _finish()
        _stop_sound(handle, gain);
        if timer then
            timer:cancel();
        end;
        assert(g_stop_funcs[horn_id] == _finish);
        g_stop_funcs[horn_id] = nil;
    end;

    local function _on_timer()
        local entry = pattern[index];

        if not (entry and (pos or (object and util.object_is_valid(object)))) then
            _finish();
            return;
        end;

        if entry.gain <= 0 then
            -- Stop sound:
            _stop_sound(handle, gain);
        elseif handle then
            -- Fade sound:
            minetest.sound_fade(handle, advtrains_horntrack.FADE_RATE, entry.gain);
            gain = entry.gain;
        else
            -- Start sound:
            local params =
            {
                object = object;
                pos    = pos;
                gain   = entry.gain;
                loop   = true;
                max_hear_distance = 64;
            };
            handle = minetest.sound_play(horn_sound, params);
            gain = entry.gain;
        end;

        index = index + 1;
        timer = minetest.after(entry.duration, _on_timer);
    end;

    local stop_func = g_stop_funcs[horn_id];
    if stop_func then
        -- Cancel current horn track pattern and play after a delay:
        stop_func();
        timer = minetest.after(advtrains_horntrack.IMPLICIT_GAP, _on_timer);
    elseif entity and entity.horn_handle then
        -- Cancel manual horn and play after a delay:
        _stop_sound(entity.horn_handle, 1.0);
        entity.horn_handle = nil;
        timer = minetest.after(advtrains_horntrack.IMPLICIT_GAP, _on_timer);
    else
        -- Play immediately:
        timer = minetest.after(0, _on_timer);
    end;
    g_stop_funcs[horn_id] = _finish;
end;


--- Parses a pattern string.
-- @param text [string]          A pattern string.
-- @return     [HornPattern|nil] The parsed pattern, or nil on failure.
-- @return     [string|nil]      A human-readable message on failure.
function advtrains_horntrack.parse_pattern(text)
    local pattern = {};
    local previous;
    for word in text:gmatch('%S+') do
        -- Parse word:
        local gain, duration = word:match('([%d.]+)[xX%*]([%d.]+)');
        if gain then
            -- '<Gain>x<Duration>':
            gain = tonumber(gain);
            duration = tonumber(duration);
        else
            -- '<Duration>':
            gain = 1;
            duration = tonumber(word);
        end;

        -- Check validity:
        if not (gain and duration) then
            return nil, S('Invalid word in horn pattern: @1.', word);
        end;

        -- If the previous entry was the same gain, insert a gap:
        if gain > 0 and previous and previous.gain > 0 and previous.gain == gain then
            table.insert(pattern, {gain = 0, duration = advtrains_horntrack.IMPLICIT_GAP});
        end;

        -- Add the entry:
        local entry = {gain = util.clamp(gain, 0, 1), duration = duration};
        if duration > 0 then
            table.insert(pattern, entry);
        end;
        previous = entry;
    end;
    return pattern, nil;
end;


--- Finds the first wagon in a train that has a horn.
-- @param train [Train]               The train to check.
-- @return      [SimpleSoundSpec|nil] The wagon's horn sound, or nil if no wagon has a horn.
-- @return      [Wagon|nil]           The wagon with the horn, or nil if no wagon has a horn.
function advtrains_horntrack.get_train_horn(train)
    for __, wagon_id in ipairs(train.trainparts) do
        local wagon = advtrains.wagons[wagon_id] or {};
        local wagon_type = advtrains.wagon_prototypes[wagon.type] or {};
        local horn = wagon_type.horn_sound;
        if horn then
            return horn, wagon;
        end;
    end;
    return nil, nil;
end;
