--[[
    Horn Track for Advanced Trains

    Copyright © 2024, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Track registration function and track definitions.
-- @module tracks


local S = advtrains_horntrack.S;
local util = advtrains_horntrack.util;


advtrains_horntrack.note_block_item = util.first_available_item('mesecons_noteblock:noteblock', 'farming:string');


--- Registers a horn track.
-- @param name        [string]    The prefixed internal name of the track node.
-- @param description [string]    The human-readable name of the track node.
-- @param base_name   [string]    The normal track node to use as a reference.
-- @param override    [table|nil] An optional table of node definition properties to override.
-- @return            [boolean]   true on success.
function advtrains_horntrack.register_horn_track(name, description, base_name, override)
    local base_def = minetest.registered_nodes[base_name];
    if not base_def then
        return false;
    end;

    local base_placer = base_def.drop;
    local base_placer_def = minetest.registered_items[base_placer];
    assert(base_placer_def, 'Failed to get track placer definition.');

    override = override or {};

    local mesh_prefix, mesh_suffix = string.match(override.mesh or base_def.mesh, '^(.*)_st(%.%w+)$');
    assert(mesh_prefix, 'Failed to extract mesh prefix and suffix.');

    local track_type = util.get_track_type(base_def);

    local function get_additional_definition(def, preset, suffix, rotation)
        local new_def = {};

        local old_after_place_node = def.after_place_node or advtrains.ndb.update;
        function new_def.after_place_node(pos)
            advtrains_horntrack.init_track(pos);
            return old_after_place_node(pos);
        end;

        function new_def.on_rightclick(pos, node, clicker, itemstack, pointed_thing)
            if minetest.is_player(clicker) then
                advtrains_horntrack.open_ui(pos, clicker);
            end;
        end;

        new_def = advtrains.merge_tables(new_def, override);
        new_def.advtrains = advtrains.merge_tables(base_def.advtrains or {}, {on_train_enter = advtrains_horntrack.on_train_enter}, override.advtrains);
        new_def.groups    = advtrains.merge_tables(base_def.groups or {}, {advtrains_horntrack = 1}, override.groups);

        return new_def;
    end;

    advtrains.register_tracks(track_type,
    {
        nodename_prefix = name;
        texture_prefix  = base_placer_def.inventory_image .. '^advtrains_horntrack';
        shared_texture  = util.modify_tile(base_def.tiles[1], '^advtrains_horntrack_overlay.png');
        models_prefix   = mesh_prefix;
        models_suffix   = mesh_suffix;
        description     = description;
        formats         = {};
        get_additional_definiton = get_additional_definition;
    },
    advtrains.trackpresets.t_30deg_straightonly);

    local placer_name = name:gsub('^:', '') .. '_placer';

    -- Crafting:
    if advtrains_horntrack.note_block_item then
        minetest.register_craft(
        {
            type   = 'shapeless';
            output = placer_name;
            recipe = {base_placer, advtrains_horntrack.note_block_item};
        });
    end;

    return true;
end;


--- The `on_train_enter` handler for horn tracks.
-- @param pos      [vector] The position of the track node.
-- @param train_id [string] The ID of the train entering the track.
-- @param train    [Train]  The train entering the track.
function advtrains_horntrack.on_train_enter(pos, train_id, train)
    local meta = minetest.get_meta(pos);
    local ars_text = meta:get_string('ars_rules');
    if not util.train_matches_ars_text(train, ars_text) then
        return;
    end;

    local pattern_text = meta:get_string('pattern');
    local pattern, message = advtrains_horntrack.parse_pattern(pattern_text);
    if not pattern then
        util.proximity_message(pos, 10, message);
        return;
    end;

    advtrains_horntrack.toot(pattern, train);
end;


--- Initialises a horn track.
-- @param pos [vector] The position of the track node.
function advtrains_horntrack.init_track(pos)
    local meta = minetest.get_meta(pos);
    meta:set_string('pattern', advtrains_horntrack.DEFAULT_PATTERN);
end;


advtrains_horntrack.register_horn_track('advtrains_horntrack:horn_track',         S'Horn Track',         'advtrains:dtrack_st');
advtrains_horntrack.register_horn_track('advtrains_horntrack:tieless_horn_track', S'Tieless Horn Track', 'advtrains:dtrack_tieless_st');
