--[[
    Horn Track for Advanced Trains

    Copyright © 2024, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Formspec functions.
-- @module ui


local S = advtrains_horntrack.S;


local FORM_NAME = 'advtrains_horntrack:horn_track';

local WIDTH = 12;
local HEIGHT = 6.75;


local pos_by_player = {};


--- Opens the horn track UI for the specified player.
-- @param pos    [vector]    The position of the track node.
-- @param player [ObjectRef] The player to open the UI for.
-- @return       [boolean]   true on success.
function advtrains_horntrack.open_ui(pos, player)
    if not minetest.is_player(player) then
        return false;
    end;

    local name = player:get_player_name();
    if minetest.is_protected(pos, name) then
        minetest.record_protection_violation(pos, name);
        return false;
    end;

    local formspec = advtrains_horntrack.create_ui(pos);
    pos_by_player[name] = pos;
    minetest.show_formspec(name, FORM_NAME, formspec);
    return true;
end;


--- Creates the track UI formspec.
-- @param pos [vector] The position of the track node.
-- @return    [string] A formspec.
function advtrains_horntrack.create_ui(pos)
    local meta = minetest.get_meta(pos);

    local formspec = {};
    local _esc = minetest.formspec_escape;
    local function _add(template, ...)
        formspec[#formspec + 1] = template:format(...);
    end;

    _add('size[%f,%f]', WIDTH, HEIGHT);
    _add('real_coordinates[true]');
    _add('set_focus[pattern;false]');

    _add('field[0.5,0.75;%f,0.75;pattern;%s;%s]', WIDTH - 1, _esc(S'Horn pattern:'), _esc(meta:get_string('pattern')));
    _add('textarea[0.5,2;%f,%f;ars_rules;%s;%s]', WIDTH - 1, HEIGHT - 3.5, _esc(S'ARS rules:'), _esc(meta:get_string('ars_rules')));

    local buttons_y = HEIGHT - 1.25;
    _add('button_exit[%f,%f;3,0.75;enter;%s]', WIDTH - 3.5, buttons_y, _esc(S'Save'));
    _add('button_exit[%f,%f;3,0.75;cancel;%s]', WIDTH - 6.75, buttons_y, _esc(S'Cancel'));
    _add('button[0.5,%f;3,0.75;test;%s]', buttons_y, _esc(S'Test'));

    return table.concat(formspec, '');
end;


--- Handles fields received from the horn track UI.
-- @param player [ObjectRef] The player.
-- @param fields [table]     Formspec fields.
function advtrains_horntrack.handle_ui_fields(player, fields)
    local pos = pos_by_player[player:get_player_name()];
    if not pos then
        return;
    end;

    local name = player:get_player_name();
    if minetest.is_protected(pos, name) then
        minetest.record_protection_violation(pos, name);
        return;
    end;

    if fields.test then
        local pattern, message = advtrains_horntrack.parse_pattern(fields.pattern);
        if pattern then
            advtrains_horntrack.toot_test(pattern, pos);
        else
            minetest.chat_send_player(name, message);
        end;
        return;
    end;

    if fields.enter or fields.key_enter then
        if fields.pattern == '' then
            fields.pattern = advtrains_horntrack.DEFAULT_PATTERN;
        end;

        local meta = minetest.get_meta(pos);
        meta:set_string('pattern', fields.pattern);
        meta:set_string('ars_rules', fields.ars_rules);
    end;
end;


minetest.register_on_player_receive_fields(
function(player, formname, fields)
    if formname == FORM_NAME then
        advtrains_horntrack.handle_ui_fields(player, fields);
    end;
end);
