--[[
    Horn Track for Advanced Trains

    Copyright © 2024-2025, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Generic utility functions.
-- @module util


--- Generic utility functions.
local util = {};
advtrains_horntrack.util = util;


--- Applies a function to every item in a table.
-- @param tbl  [table]    A table.
-- @param func [function] A function to call for each value.
-- @return     [table]    A new table with the same keys as `tbl`.
function util.map(tbl, func)
    local result = {};
    for key, value in pairs(tbl) do
        result[key] = func(value);
    end;
    return result;
end;


--- Applies a texture suffix to a tile definition, preserving animation and other attributes.
-- @param tile [string|table] A tile definition.
-- @param mod  [string]       A texture modifier, which should start with '^'.
-- @return     [string|table] A modified tile definition.
function util.modify_tile(tile, mod)
    if type(tile) == 'string' then
        return tile .. mod;
    else
        return advtrains.merge_tables(tile, {name = (tile.name or tile.image) .. mod});
    end;
end;


--- Checks if the train matches the ARS rules.
-- This function always returns true if Advanced Trains Interlocking is unavailable.
-- @param train    [Train]   The train to check.
-- @param ars_text [string]  A set of ARS rules in string format.
-- @return         [boolean] true if the train matches the ARS rules.
function util.train_matches_ars_text(train, ars_text)
    if not advtrains.interlocking then
        return true;
    end;

    if ars_text == '' or ars_text == '*' then
        return true;
    end;

    local ars = advtrains.interlocking.text_to_ars(ars_text);
    return advtrains.interlocking.ars_check_rule_match(ars, train);
end;


--- Sends a chat message to all players near a position.
-- @param pos     [vector] The centre of the sphere.
-- @param range   [number] The radius of the sphere.
-- @param message [string] The message to send.
function util.proximity_message(pos, range, message)
    for __, player in ipairs(core.get_connected_players()) do
        if player:get_pos():distance(pos) <= range then
            core.chat_send_player(player:get_player_name(), message);
        end;
    end;
end;


--- Clamps a number to within a specified range.
-- @param value [number] The value to clamp.
-- @param min   [number] The lower bound.
-- @param max   [number] The upper bound.
-- @return      [number] min, max, or value.
function util.clamp(value, min, max)
    if value < min then
        return min;
    elseif value > max then
        return max;
    else
        return value;
    end;
end;


--- Returns the first item available for crafting.
-- Groups are always considered to be available.
-- @param ... [string]     Item names.
-- @return    [string|nil] The name of the first available item, or nil if none were available.
function util.first_available_item(...)
    for __, name in ipairs({...}) do
        if core.registered_items[name] or string.match(name, '^group:.*') then
            return name;
        end;
    end;
    return nil;
end;


--- Takes a track node definition and determines its track type.
-- @param track_def [table]  A track node definition.
-- @return          [string] The node's track type.
function util.get_track_type(track_def)
    local groups = track_def.groups or {};
    local types = {};
    for group, level in pairs(groups) do
        if level > 0 then
            local track_type = group:match('^advtrains_track_(.+)$');
            if track_type then
                table.insert(types, track_type);
            end;
        end;
    end;
    if #types == 1 then
        return types[1];
    end;
    error(('Cannot determine track type for %q.'):format(track_def.name));
end;


--- Checks if an object reference is valid.
-- This is equivalent to `object:is_valid()`, but works on older versions of Luanti.
-- @param object [ObjectRef] The object to check.
-- @return       [boolean]   true if the object reference is valid.
function util.object_is_valid(object)
    if object.is_valid then
        return object:is_valid();
    else
        return object:get_pos() ~= nil;
    end;
end;
