--[[
    Deepslate — adds deepslate and related building materials to Minetest
    Copyright © 2022‒2023, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Provides public API functions.


local S = deepslate.S;


deepslate.ores = {};
deepslate.conversion_mapping = {};
deepslate.conversion_mapping_param2 = {};


--- Registers a node to be converted during deepslate generation.
-- @param from   The name of the node to replace.
-- @param to     The name of the node to replace it with.
-- @param param2 The param2 value for the new node, or `deepslate.PRESERVE_PARAM2` to keep the old node's param2.
-- @return       true if the conversion was registered, or false if either node was unavailable.
function deepslate.register_conversion(from, to, param2)
    if not minetest.registered_nodes[from] then
        return false;
    elseif not minetest.registered_nodes[to] then
        minetest.log('warning', ('[Deepslate] Failed to register conversion from %q: %q doesn\'t exist.'):format(from, to));
        return false;
    end;

    deepslate.conversion_mapping[from] = to;
    deepslate.conversion_mapping_param2[from] = param2 or 0;
    return true;
end;


local function extract_ore_texture(tile)
    if type(tile) ~= 'string' then
        tile = tile.name;
    end;
    return string.match(tile, '^default_stone%.png%^(.*)');
end;


--- Registers a deepslate version of an ore node.
-- @param name     The namespaced ID of the deepslate ore node to register.
-- @param ore_name The namespaced IF of the original ore node.
-- @param options  A table of additional options: ore_texture.
-- @param override A partial node definition table, which overrides the generated definition.
-- @return         true if the ore node was defined, or false if the original ore is not available.
function deepslate.register_ore(name, ore_name, options, override)
    local ore_def = minetest.registered_nodes[ore_name];
    if not ore_def then
        return false;
    end;

    options  = options  or {};
    override = override or {};

    -- Groups:
    local groups = table.copy(ore_def.groups or {});
    groups.deepslate = 1;
    groups.pickaxey  = groups.pickaxey or groups.level or 3;
    groups.level     = deepslate.settings.level + (groups.level or 0);
    for key, value in pairs(override.groups or {}) do
        groups[key] = value;
    end;

    -- Textures:
    local function _create_texture(stone, ore)
        assert(ore, ('[Deepslate] Could not extract overlay texture from %q.'):format(ore_name));
        if string.find(stone, '[()^]') or string.find(ore, '[()^]') then
            -- Complex textures — explicitly scale to a constant size:
            local size = 256;
            return string.format('((%s)^[resize:256x256)^((%s)^[resize:%dx%d)', stone, ore, size, size);
        else
            -- Simple textures — implicitly scale to the largest size:
            return string.format('%s^%s', stone, ore);
        end;
    end;
    local ore_texture  = options.ore_texture or extract_ore_texture(ore_def.tiles[1]);
    local top_texture  = _create_texture('mcl_deepslate_top.png', ore_texture);
    local side_texture = _create_texture('mcl_deepslate.png',     ore_texture);

    -- Drops:
    local drop = ore_def.drop;
    if drop == ore_name or (type(drop) == 'table' and drop.name == ore_name) then
        drop = name;
    end;

    -- Definition:
    local def =
    {
        description       = S('Deepslate @1', ore_def.description);
        tiles             = {top_texture, top_texture, side_texture};
        sounds            = ore_def.sounds or deepslate.sounds.node_sound_stone_defaults();
        groups            = groups;
        drop              = drop;
        light_source      = ore_def.light_source;
        is_ground_content = true;
        _mcl_hardness     = 4.5;
    };

    -- Overrides:
    for key, value in pairs(override) do
        def[key] = value;
    end;

    minetest.register_node(name, def);

    table.insert(deepslate.ores, name);
    deepslate.register_conversion(ore_name, name);

    -- Smelting:
    local result = minetest.get_craft_result{method = 'cooking', items = {ore_name}};
    if result.time > 0 then
        minetest.register_craft(
        {
            type   = 'cooking';
            output = result.item:to_string();
            recipe = name;
            time   = result.time;
        });
    end;

    return true;
end;
