--[[
    Leads — Adds leads for transporting animals to Minetest.
    Copyright © 2023, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Knot entity definition.
-- @module knots


local S = leads.S;


--- Ties a lead to a post.
-- @type KnotEntity
leads.KnotEntity = {};

leads.KnotEntity.description = S'Lead Knot';

leads.KnotEntity._leads_immobile  = true;
leads.KnotEntity._leads_leashable = true;

leads.KnotEntity.initial_properties =
{
    visual          = 'mesh';
    visual_size     = vector.new(10, 10, 10);
    mesh            = 'leads_lead_knot.obj';
    textures        = {'leads_lead_knot.png'};
    physical        = false;
    selectionbox    = {-3/16, -4/16, -3/16, 3/16, 4/16, 3/16};
};

--- Spawns or loads a knot.
function leads.KnotEntity:on_activate(staticdata, dtime_s)
    self.num_connections = 0;

    local data = minetest.deserialize(staticdata);
    if data then
        self.num_connections = data.num_connections or 0;
    end;
end;

--- Steps the knot.
function leads.KnotEntity:on_step(dtime, moveresult)
    if self.num_connections <= 0 then
        self.object:remove();
    end;
end;

--- Returns the knot's state as a table.
function leads.KnotEntity:get_staticdata()
    local data = {num_connections = self.num_connections};
    return minetest.serialize(data);
end;

--- Handles the knot being punched.
function leads.KnotEntity:on_punch(puncher, time_from_last_punch, tool_capabilities, dir, damage)
    -- Transfer all connected leads to the puncher:
    if puncher and not puncher:get_player_control().sneak then
        self:transfer_leads(puncher);
    end;

    -- Remove this knot:
    self.object:remove();
    return true;
end;

--- Transfers all leads attached to this knot to another object.
-- @param leader [ObjectRef] The new leader.
function leads.KnotEntity:transfer_leads(leader)
    for lead, is_leader in leads.find_connected_leads(self.object, true, true) do
        local entity = lead:get_luaentity();
        if not is_leader then
            entity:reverse();
        end;
        entity:set_leader(leader);
    end;
end;

--- Handles the knot being right-clicked.
function leads.KnotEntity:on_rightclick(clicker)
    leads.knot(clicker, self.object:get_pos());
end;

--- Called when a lead is added.
function leads.KnotEntity:_leads_lead_add(lead, is_leader)
    self.num_connections = self.num_connections + 1;
end;

--- Called when a lead is removed.
function leads.KnotEntity:_leads_lead_remove(lead, is_leader)
    self.num_connections = self.num_connections - 1;
end;

minetest.register_entity('leads:knot', leads.KnotEntity);
