--[[
    Rivets — Adds rivets to Minetest
    Written in 2023 by Silver Sandstone <@SilverSandstone@craftodon.social>

    To the extent possible under law, the author has dedicated all copyright
    and related and neighbouring rights to this software to the public
    domain worldwide. This software is distributed without any warranty.

    You should have received a copy of the CC0 Public Domain Dedication
    along with this software. If not, see
    <https://creativecommons.org/publicdomain/zero/1.0/>.
]]


--- Rivets
-- @module rivets


local S = minetest.get_translator('rivets');


rivets = {};

rivets.registered_rivets = {};


rivets.NODEBOX =
{
    type = 'fixed';
    fixed = {-2/16, -8/16, -2/16,
              2/16, -6/16,  2/16};
};


--- Creates a colour texture from a dark shade and a light shade.
-- @tparam  string         name   The base name of the texture, without the extension.
-- @tparam  ColorSpec|nil  dark   The dark colour.
-- @tparam  ColorSpec|nil  light  The light colour.
-- @treturn string                A texture string.
function rivets.create_texture(name, dark, light)
    if not (dark and light) then
        return name .. '.png';
    end;
    dark  = minetest.colorspec_to_colorstring(dark);
    light = minetest.colorspec_to_colorstring(light);
    return ('(%s_base.png^[multiply:%s)^((%s_mask.png^[mask:%s_base.png)^[multiply:%s)'):format(name, dark, name, name, light);
end;


--- Checks if rivets can be placed on the specified node.
-- @tparam  string|Node  node  The node name or table.
-- @treturn boolean            `true` if the node can be riveted.
function rivets.can_rivet(node)
    local name = node;
    if type(name) ~= 'string' then
        name = node.name;
    end;

    local def = minetest.registered_nodes[name];
    if not def then
        return false;
    end;

    if def.can_rivet ~= nil then
        return def.can_rivet;
    end;

    if not def.walkable then
        return false;
    end;

    local drawtype = string.match(def.drawtype or 'normal', '%w+');
    if drawtype == 'normal' or drawtype == 'glasslike' or drawtype == 'allfaces' then
        return true;
    end;

    return false;
end;


--- Checks if the node at the specified position is riveted on any side.
-- @tparam  vector  pos  The position of the node to check.
-- @treturn boolean      `true` if the node is riveted.
function rivets.node_is_riveted(pos)
    for wallmounted = 0, 5 do
        local offset = minetest.wallmounted_to_dir(wallmounted);
        local rivet_node = minetest.get_node(pos - offset);
        local rivet_def = minetest.registered_nodes[rivet_node.name];
        if rivet_def and rivet_def._rivet_material then
            if rivet_node.param2 == wallmounted then
                return true;
            end;
        end;
    end;
    return false;
end;


local old_is_protected = minetest.is_protected;
function minetest.is_protected(pos, name)
    return old_is_protected(pos, name) or rivets.node_is_riveted(pos);
end;


--- The on_place handler for rivets.
-- @tparam  ItemStack     itemstack      The item the player is holding.
-- @tparam  ObjectRef     placer         The player placing the rivet.
-- @tparam  PointedThing  pointed_thing  The pointed-thing.
-- @treturn ItemStack|nil                The leftover items.
function rivets.on_place(itemstack, placer, pointed_thing)
    local under_pos = pointed_thing.under;
    if not under_pos then
        return minetest.item_place(itemstack, placer, pointed_thing);
    end;

    if minetest.is_player(player) and not player:get_player_control().sneak then
        local under_node = minetest.get_node(under_pos);
        local under_def = minetest.registered_nodes[under_node.name];
        local on_rightclick = under_def.on_rightclick;
        if on_rightclick then
            return on_rightclick(pointed_thing.under, under_node, placer, itemstack, pointed_thing) or itemstack;
        end;
    end;

    local under_node = minetest.get_node(pointed_thing.under);
    if not rivets.can_rivet(under_node) then
        return nil;
    end;

    return minetest.item_place(itemstack, placer, pointed_thing);
end;


--- Registers a rivet type.
-- @tparam  string         name         The prefixed item ID of the rivet node.
-- @tparam  string         ingot_name   The prefixed item ID of the ingot to make the rivet out of.
-- @tparam  string         description  The translatable description of the rivet.
-- @tparam  ColorSpec|nil  dark_shade   The darker colour to use for the rivet's textures.
-- @tparam  ColorSpec|nil  light_shade  The lighter colour to use for the rivet's textures.
-- @tparam  table|nil      override     A table of arbitrary values to override the node definition.
-- @treturn boolean                     `true` if the rivet was registered, or false if it was skipped.
function rivets.register_rivet(name, ingot_name, description, dark_shade, light_shade, override)
    local ingot_def = minetest.registered_items[ingot_name];
    if not ingot_def then
        return false;
    end;

    rivets.registered_rivets[ingot_name] = name;

    local groups =
    {
        rivet       = 1;
        cracky      = 1;
        level       = 1;
        pickaxey    = 2;
    };

    local texture = rivets.create_texture('rivets_rivet', dark_shade, light_shade);
    local icon = rivets.create_texture('rivets_inv', dark_shade, light_shade);

    local def =
    {
        description         = description;
        paramtype           = 'light';
        paramtype2          = 'wallmounted';
        drawtype            = 'mesh';
        mesh                = 'rivets_rivet.obj';
        selection_box       = rivets.NODEBOX;
        collision_box       = rivets.NODEBOX;
        tiles               = {texture};
        inventory_image     = icon;
        wield_image         = icon;
        groups              = groups;
        on_place            = rivets.on_place;
        sunlight_propagates = true;
        is_ground_content   = false;
        walkable            = false;
        _rivet_material     = ingot_name;
    };

    for key, value in pairs(override or {}) do
        def[key] = value;
    end;

    minetest.register_node(name, def);

    -- Crafting recipe:
    minetest.register_craft(
    {
        output = name .. ' 25';
        recipe =
        {
            {ingot_name, ingot_name, ingot_name},
            {'',         ingot_name, ''},
            {'',         ingot_name, ''},
        };
    });

    if minetest.get_modpath('rp_crafting') then
        crafting.register_craft{items = {ingot_name}, output = name .. ' 5'};
    end;

    return true;
end;


-- Minetest Game:
rivets.register_rivet('rivets:steel_rivet',         'default:steel_ingot',              S'Steel Rivet',         '#555555', '#DDDDDD');
rivets.register_rivet('rivets:bronze_rivet',        'default:bronze_ingot',             S'Bronze Rivet',        '#7B3004', '#FA761D');

-- MineClone:
rivets.register_rivet('rivets:iron_rivet',          'mcl_core:iron_ingot',              S'Iron Rivet',          '#555555', '#DDDDDD');
rivets.register_rivet('rivets:netherite_rivet',     'mcl_nether:netherite_ingot',       S'Netherite Rivet',     '#111111', '#747274');

-- Repixture:
rivets.register_rivet('rivets:wrought_iron_rivet',  'rp_default:ingot_wrought_iron',    S'Wrought Iron Rivet',  '#666666', '#CCCCCC');
rivets.register_rivet('rivets:steel_rivet',         'rp_default:ingot_steel',           S'Steel Rivet',         '#555555', '#DDDDDD');
rivets.register_rivet('rivets:carbon_steel_rivet',  'rp_default:ingot_carbon_steel',    S'Carbon Steel Rivet',  '#657186', '#B7C8E9');
rivets.register_rivet('rivets:bronze_rivet',        'rp_default:ingot_bronze',          S'Bronze Rivet',        '#7B3004', '#FA761D');

-- Basic Materials:
rivets.register_rivet('rivets:brass_rivet',         'basic_materials:brass_ingot',      S'Brass Rivet',         '#78570C', '#E3D73D');

-- Titanium:
rivets.register_rivet('rivets:titanium_rivet',      'titanium:titanium',                S'Titanium Rivet',      '#3A2D2D', '#D0BBBB');
