--[[
    Repixture Crafting Compatibility — Converts crafting recipes from Minetest to Repixture.
    Written in 2022‒2023 by Silver Sandstone <@SilverSandstone@craftodon.social>

    To the extent possible under law, the author has dedicated all copyright
    and related and neighbouring rights to this software to the public
    domain worldwide. This software is distributed without any warranty.

    You should have received a copy of the CC0 Public Domain Dedication
    along with this software. If not, see
    <https://creativecommons.org/publicdomain/zero/1.0/>.
]]


--- Repixture Crafting Compatibility
--
-- NOTE: This project uses the term ‘registration format’ to refer to the
-- recipe format passed to `minetest.register_craft()`, and ‘alternate format’
-- to refer to the format returned from `minetest.get_all_craft_recipes()`.
-- Some of these functions accept either format.


rp_craftcompat = {};

rp_craftcompat.queue = {};


--- Lists all registered crafting recipes.
-- @return A sequence of crafting recipes in alternate format.
function rp_craftcompat.list_all_recipes()
    local result = {};
    for name in pairs(minetest.registered_items) do
        for __, recipe in ipairs(minetest.get_all_craft_recipes(name) or {}) do
            table.insert(result, recipe);
        end;
    end;
    return result;
end;


--- Checks if the specified node/item has been registered.
-- The empty hand ('') and groups ('group:*') are always considered to exist.
-- @param name The prefixed item name.
-- @return true if the item exists.
function rp_craftcompat.item_exists(name)
    return name == ''
        or minetest.registered_items[name] ~= nil
        or name:match('^group:') ~= nil;
end;


--- Checks if the specified recipe can be converted to Repixture.
-- @param recipe The recipe, in registration format or alternate format.
-- @return true if the recipe can be converted.
function rp_craftcompat.is_convertable(recipe)
    if recipe._rp_craftcompat_disable then
        return false;
    end;

    return recipe.type == 'normal'
        or recipe.type == 'shaped'
        or recipe.type == 'shapeless'
        or recipe.type == nil;
end;


--- Checks if two lists have the same values, but not necessarily in the same order.
-- @param set1 The first list to compare.
-- @param set2 The second list to compare.
-- @return true if the lists have the same items.
function rp_craftcompat.sets_equal(set1, set2)
    local function _set_to_counts(set)
        local counts = {};
        for __, value in ipairs(set) do
            counts[value] = (counts[value] or 0) + 1;
        end;
        return counts;
    end;

    local counts1 = _set_to_counts(set1);
    local counts2 = _set_to_counts(set2);
    return rp_craftcompat.tables_equal(counts1, counts2);
end;


--- Checks if two mapping tables are equal.
-- @param table1 The first table to compare.
-- @param table2 The second table to compare.
-- @return true if the tables are the same.
function rp_craftcompat.tables_equal(table1, table2)
    -- Check for items that have been removed or changed:
    for key, value in pairs(table1) do
        if value ~= table2[key] then
            return false;
        end;
    end;

    -- Check for items that have been added:
    for key in pairs(table2) do
        if table1[key] == nil then
            return false;
        end;
    end;

    return true;
end;


--- Converts a recipe to Repixture format.
-- @param recipe The recipe to convert, in registration format or alternate format.
-- @return The same recipe, in Repixture format.
function rp_craftcompat.convert_recipe(recipe)
    if not rp_craftcompat.is_convertable(recipe) then
        return nil;
    end;

    -- Convert the recipe to a linear list of single items with duplicates:
    local counts = {};
    local ingredients = {};

    local function _process_items(items)
        for __, item in pairs(items) do
            if type(item) == 'table' then
                _process_items(item);
            elseif item ~= nil then
                table.insert(ingredients, ItemStack(item));
            end;
        end;
    end;
    _process_items{recipe.items, recipe.recipe, recipe._rp_craftcompat_items};

    -- Convert the recipe to a table of {name: count}:
    for __, stack in pairs(ingredients) do
        local name = stack:get_name();
        if not rp_craftcompat.item_exists(name) then
            return nil;
        elseif name ~= '' then
            counts[name] = (counts[name] or 0) + stack:get_count();
        end;
    end;

    -- Convert the count mapping to a linear list of unique items with counts:
    local items = {};
    for item, count in pairs(counts) do
        local stack = ItemStack(item);
        stack:set_count(count);
        table.insert(items, stack:to_string());
    end;
    table.sort(items);

    return {items = items, output = recipe.output};
end;


--- Checks if a recipe already exists.
-- @param recipe The recipe to check for, in Repixture format.
-- @return true if the exact recipe has already been registered.
function rp_craftcompat.recipe_exists(recipe)
    for __, this_recipe in ipairs(crafting.registered_crafts) do
        if this_recipe.output_str == recipe.output then
            local items = {};
            for __, item in ipairs(this_recipe.items) do
                if type(item) ~= 'string' then
                    item = item:to_string();
                end;
                table.insert(items, item);
            end;
            if rp_craftcompat.sets_equal(items, recipe.items) then
                return true;
            end;
        end;
    end;
    return false;
end;


--- A helper function to register a crafting recipe.
-- Registers the recipe to both Minetest and Repixture.
-- This function can be used as a drop-in replacement for
-- `minetest.register_craft()`.
-- @param recipe A crafting recipe in registration format.
function rp_craftcompat.register_craft(recipe)
    assert(rp_craftcompat.queue, 'rp_craftcompat.register_craft() should only be called during mod initialisation.');
    minetest.register_craft(recipe);
    table.insert(rp_craftcompat.queue, recipe);
end;


function rp_craftcompat.on_mods_loaded()
    if minetest.settings:get_bool('rp_craftcompat.auto_convert') then
        table.insert_all(rp_craftcompat.queue, rp_craftcompat.list_all_recipes());
    end;

    for __, recipe in ipairs(rp_craftcompat.queue) do
        local rp_recipe = rp_craftcompat.convert_recipe(recipe);
        if rp_recipe and not rp_craftcompat.recipe_exists(rp_recipe) then
            crafting.register_craft(rp_recipe);
        end;
    end;

    rp_craftcompat.queue = nil;
end;

minetest.register_on_mods_loaded(rp_craftcompat.on_mods_loaded);
