--[[
    Subtitles — adds subtitles to Minetest.

    This module provides the SubtitleDisplay class, which is the base
    class for subtitle displays.

    Copyright © 2022, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


subtitles.SubtitleDisplay = subtitles.Object:extend();

subtitles.SubtitleDisplay.implementations = {};

subtitles.SubtitleDisplay.NAME = nil;
subtitles.SubtitleDisplay.ICON = 'subtitles_mode_unknown.png';
subtitles.SubtitleDisplay.TITLE = nil;
subtitles.SubtitleDisplay.DESCRIPTION = nil;

function subtitles.SubtitleDisplay:new(username)
    self.username = username;
    self.sounds = {};
    self._timers = {};

    self:init();
end;

function subtitles.SubtitleDisplay:init()
end;

function subtitles.SubtitleDisplay:register()
    assert(self.NAME);
    table.insert(subtitles.SubtitleDisplay.implementations, self);
end;

function subtitles.SubtitleDisplay:handle_sound_play(sound)
    if sound:is_exempt() then
        return;
    end;

    self.sounds[sound.handle or sound] = sound;
    local duration = sound:get_duration();
    if duration then
        self._timers[sound] = minetest.after(duration, function() self:handle_sound_stop(sound); end);
    end;
    self:add(sound);
end;

function subtitles.SubtitleDisplay:handle_sound_stop(sound_or_handle)
    local sound;
    if type(sound_or_handle) == 'table' then
        sound = sound_or_handle;
    else
        sound = self.sounds[sound_or_handle];
    end;

    if not sound then
        return;
    end;
    local job = self._timers[sound];
    if job then
        job:cancel();
    end;
    self.sounds[sound.handle or sound] = nil;
    self:remove(sound);
end;

function subtitles.SubtitleDisplay:destroy()
    for __, sound in pairs(self.sounds) do
        self:handle_sound_stop(sound);
    end;
    self.sounds = {};
end;

function subtitles.SubtitleDisplay:__tostring()
    return ('[Subtitle display %q for player %q]'):format(self.NAME, self.username);
end;

function subtitles.SubtitleDisplay:get_player()
    return minetest.get_player_by_name(self.username);
end;

function subtitles.SubtitleDisplay:add(sound, message)
    error('Not implemented.');
end;

function subtitles.SubtitleDisplay:remove(sound)
    error('Not implemented.');
end;

function subtitles.SubtitleDisplay:step(dtime)
end;

-- Static methods:

function subtitles.SubtitleDisplay.get_by_name(name)
    for __, impl in ipairs(subtitles.SubtitleDisplay.implementations) do
        if impl.NAME == name then
            return impl;
        end;
    end;
    return nil;
end;
