--[[
    Subtitles — adds subtitles to Minetest.

    This module provides public API functions.

    Copyright © 2022, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


local S = subtitles.S;


subtitles.registered_descriptions = {};
subtitles.registered_parameters = {};
subtitles.agents_by_player = {};
subtitles.reported_missing = {};


subtitles.DEFAULT_DURATION = 3;
subtitles.EPHEMERAL_DURATION = 1;
subtitles.FOOTSTEP_DURATION = 0.5;
subtitles.DEFAULT_MAX_HEAR_DISTANCE = 32;
subtitles.FOOTSTEP_HEAR_DISTANCE = 16;
subtitles.FOOTSTEP_INTERVAL = 0.25;


function subtitles.on_sound_play(spec, parameters, handle)
    local sound = subtitles.Sound(spec, parameters, handle);
    if sound:is_exempt() then
        return;
    end;
    for __, player in ipairs(sound:get_players()) do
        local agent = subtitles.get_agent(player);
        if agent:get_enabled() and sound:is_in_range_of_player(player) then
            local display = agent:get_display();
            display:handle_sound_play(sound);
        end;
    end;
end;


function subtitles.on_sound_stop(handle)
    for username, agent in pairs(subtitles.agents_by_player) do
        agent:get_display():handle_sound_stop(handle);
    end;
end;


function subtitles.get_agent(player)
    if minetest.is_player(player) then
        player = player:get_player_name();
    end;

    local agent = subtitles.agents_by_player[player];
    if not agent then
        agent = subtitles.Agent(player);
        subtitles.agents_by_player[player] = agent;
    end;
    return agent;
end;


function subtitles.on_node_action(pos, node, action)
    if type(node) ~= 'string' then
        node = node.name;
    end;

    local spec = subtitles.util.get_node_sound(node, action);
    if spec then
        local parameters = {pos = pos, duration = subtitles.EPHEMERAL_DURATION};
        subtitles.on_sound_play(spec, parameters, nil);
    end;
end;


function subtitles.register_description(name, description, parameters)
    if description then
        subtitles.registered_descriptions[name] = description;
    end;
    if parameters then
        local old_params = subtitles.registered_parameters[name] or {};
        subtitles.registered_parameters[name] = subtitles.util.update(old_params, parameters);
    end;
end;


function subtitles.report_missing(name)
    if not subtitles.reported_missing[name] then
        subtitles.reported_missing[name] = true;
        minetest.log('warning', ('[Subtitles] No description available for sound ‘%s’.'):format(name));
    end;
end;


function subtitles.show_intro(player)
    local command = minetest.colorize('#FFFFBF', '/subtitles');
    local message = S('Subtitles are available on this server. Type ‘@1’ to manage your preferences.', command);
    minetest.chat_send_player(player:get_player_name(), message);
end;
