--[[
    Subtitles — adds subtitles to Minetest.

    This module provides the BaseTextSubtitleDisplay class, a base class
    for subtitle displays using text HUDs.

    Copyright © 2022, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


local S = subtitles.S;


subtitles.BaseTextSubtitleDisplay = subtitles.SubtitleDisplay:extend();

function subtitles.BaseTextSubtitleDisplay:init()
    self.entries_by_slot  = {};
    self.entries_by_sound = {};
    self.entries_by_merge = {};

    self.margin    = 56;
    self.size      = {x = 360, y = 40};
    self.font_size = 2.0;
    self.style     = subtitles.util.TextStyle.REGULAR;
    self.position  = {x = 1, y = 1};
    self.alignment = {x = 0, y = -1};
    self.z_index   = 100;
    self.spacing   = 0;
    self.direction = -1;
    self.offset    = {x = 0, y = 0};
end;

function subtitles.BaseTextSubtitleDisplay:add(sound)
    local player = self:get_player();
    if not player then
        return;
    end;

    local merge_key = sound:get_merge_key();
    local entry = merge_key and self.entries_by_merge[merge_key];
    if entry then
        entry:ref();
    else
        local slot = self:get_free_slot();
        entry = self.Entry(self, sound, slot);
        entry:show();
        self.entries_by_slot[slot]   = entry;
    end;
    self.entries_by_sound[sound] = entry;
    if merge_key then
        self.entries_by_merge[merge_key] = entry;
    end;
end;

function subtitles.BaseTextSubtitleDisplay:remove(sound)
    local entry = self.entries_by_sound[sound];
    if not entry then
        return;
    end;
    entry:unref();
    self.entries_by_sound[sound] = nil;
end;

function subtitles.BaseTextSubtitleDisplay:remove_entry(entry)
    self.entries_by_slot[entry.slot] = nil;
    local merge_key = entry.sound:get_merge_key();
    if merge_key then
        self.entries_by_merge[merge_key] = nil;
    end;
end;

function subtitles.BaseTextSubtitleDisplay:get_free_slot()
    local slot = 0;
    while self.entries_by_slot[slot] do
        slot = slot + 1;
    end;
    return slot;
end;

function subtitles.BaseTextSubtitleDisplay:get_huds()
    error('Not implemented.');
end;

function subtitles.BaseTextSubtitleDisplay:get_hud_defaults(entry)
    local hud = {};
    hud.position  = self.position;
    hud.alignment = self.alignment;
    hud.z_index   = self.z_index;
    hud.direction = subtitles.util.HUDDirection.LEFT_TO_RIGHT;
    return hud;
end;


subtitles.BaseTextSubtitleDisplay.Entry = subtitles.Object:extend();

function subtitles.BaseTextSubtitleDisplay.Entry:new(display, sound, slot)
    self.display = display;
    self.sound   = sound;
    self.slot    = slot;

    self.ref_count = 1;
    self.huds = {};
    self.hud_ids = {};
    self.hud_defaults = self.display:get_hud_defaults(self);
end;

function subtitles.BaseTextSubtitleDisplay.Entry:show()
    local player = self.display:get_player()
    if not player then
        return;
    end;

    --[[
    local margin = self.display.margin;
    local offset = {x = -(margin + self.display.size.x / 2), y = -(margin + self.slot * self.display.size.y)};]]

    local huds = self.display:get_huds(self);
    for key, hud in pairs(huds) do
        local old_hud = self.huds[key];
        if old_hud then
            local hud_id = self.hud_ids[key];
            subtitles.util.update_hud(player, hud_id, old_hud, hud);
        else
            hud = subtitles.util.update(self.hud_defaults, hud);
            self.hud_ids[key] = player:hud_add(hud);
            self.huds[key] = hud;
        end;
    end;
end;

function subtitles.BaseTextSubtitleDisplay.Entry:hide()
    local player = self.display:get_player();
    if player then
        for __, hud_id in pairs(self.hud_ids) do
            player:hud_remove(hud_id);
        end;
    end;
    self.hud_ids = {};
    self.huds = {};
end;

function subtitles.BaseTextSubtitleDisplay.Entry:get_offset()
    local base = self.display.offset;
    local offset =
    {
        x = base.x;
        y = base.y + (self.slot - 1) * (self.display.size.y + self.display.spacing) * self.display.direction;
    };
    if self.display.direction < 0 then
        offset.y = offset.y - self.display.size.y;
    end;
    return offset;
end;

function subtitles.BaseTextSubtitleDisplay.Entry:ref()
    self.ref_count = self.ref_count + 1;
end;

function subtitles.BaseTextSubtitleDisplay.Entry:unref()
    self.ref_count = self.ref_count - 1;
    if self.ref_count <= 0 then
        self:hide()
        self.display:remove_entry(self);
    end;
end;
