--[[
    Subtitles — adds subtitles to Minetest.

    This module provides generic utility functions.

    Copyright © 2022, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


local S = subtitles.S;


subtitles.util = {};


subtitles.util.HUDDirection =
{
    LEFT_TO_RIGHT = 0;
    RIGHT_TO_LEFT = 1;
    TOP_TO_BOTTOM = 2;
    BOTTOM_TO_TOP = 3;
};

subtitles.util.TextStyle =
{
    REGULAR   = 0;
    BOLD      = 1;
    ITALIC    = 2;
    MONOSPACE = 4;
};


function subtitles.util.get_node_sound(name, keys)
    if type(keys) == 'string' then
        keys = {keys};
    end;

    -- Sounds from node definition:
    local def = minetest.registered_nodes[name];
    local sounds = def and def.sounds;
    if sounds then
        for __, key in ipairs(keys) do
            local sound = sounds[key];
            if sound then
                return sound;
            end;
        end;
    end;

    -- TODO: Get node sounds based on groups.

    return nil;
end;


function subtitles.util.describe_damage(reason)
    if reason.type == 'fall' then
        return S('Hits ground');
    elseif reason.type == 'punch' then
        return S('Punched');
    elseif reason.type == 'drown' then
        return S('Drowning');
    else
        return S('Person hurts');
    end;
end;


function subtitles.util.update(tbl, ...)
    for __, updates in ipairs{...} do
        for key, value in pairs(updates) do
            tbl[key] = value;
        end;
    end;
    return tbl;
end;


function subtitles.util.object_is_walking(objref)
    if objref:get_player_control().sneak then
        return false;
    end;

    local velocity = objref:get_velocity();
    return velocity and vector.length(velocity) >= 0.99;
end;


function subtitles.util.get_feet_pos(objref)
    local pos = objref:get_pos();
    local properties = objref:get_properties();
    return pos:offset(0, properties.collisionbox[2], 0);
end;


function subtitles.util.update_hud(player, hud_id, old_def, new_def)
    for key, value in pairs(new_def) do
        if value ~= old_def[key] then
            player:hud_change(hud_id, key, value);
        end;
    end;
end;
