--[[
    Subtitles — adds subtitles to Minetest.

    Copyright © 2022, Silver Sandstone <@SilverSandstone@craftodon.social>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
]]


--- Implements the subtitle menu GUI.


local S = subtitles.S;


--- Creates a menu formspec for the specified player.
-- @param player The username or ObjectRef of the player.
-- @param embedded This should be true if the menu is being embedded in another formspec.
-- @param options A table of extra options: width, height.
-- @return A formspec string.
function subtitles.get_menu_formspec(player, embedded, options)
    local agent = subtitles.get_agent(player);
    player = agent.username;
    options = options or {};

    local _esc = minetest.formspec_escape;
    local width = options.width or 8;
    local height = options.height or 3.75 + #subtitles.SubtitleDisplay.implementations * 1.25;
    local enabled = agent:get_enabled();
    local display_name = agent:get_display_name();

    local state_icon = 'subtitles_icon.png';
    local state_label;
    local toggle_tooltip;
    if enabled then
        state_label = S'Subtitles enabled';
        toggle_tooltip = S'Click to disable subtitles';
    else
        state_label = S'Subtitles disabled';
        toggle_tooltip = S'Click to enable subtitles';
        state_icon = state_icon .. '^subtitles_slash.png';
    end;

    -- Main menu:
    local formspec =
    {
        ('size[%f,%f] real_coordinates[true]'):format(width, height),
        ('image_button[0.25,0.25;1,1;%s;toggle;]'):format(_esc(state_icon)),
        ('tooltip[toggle;%s]'):format(_esc(toggle_tooltip)),
        ('label[1.5,0.75;%s]'):format(_esc(state_label)),

        ('image[0.25,2.0;%f,%f;subtitles_panel.png;2]'):format(width - 0.5, height - 3.5),
        ('label[0.25,1.75;%s]'):format(_esc(S'Display mode:')),

        ('button_exit[%f,%f;3,1;done;%s]'):format(width - 3.25, height - 1.25, _esc(S'Done')),
    };
    if embedded then
        formspec[1] = '';
    end;

    -- Display modes:
    for index, impl in ipairs(subtitles.SubtitleDisplay.implementations) do
        local y = 1 + index * 1.25;
        table.insert_all(formspec,
        {
            ('image[0.5,%f;1,1;%s]'):format(y, _esc(impl.ICON)),
            ('button[1.75,%f;%f,1;display_%s;%s]'):format(y, width - 3.5, impl.NAME, _esc(impl.TITLE)),
            ('tooltip[display_%s;%s]'):format(impl.NAME, _esc(impl.DESCRIPTION or impl.TITLE)),
        });
        if impl.NAME == display_name then
            local colour = enabled and '#00FF00' or '#BFBFBF';
            table.insert(formspec, ('image[%f,%f;1,1;subtitles_circle.png^[multiply:%s]'):format(width - 1.5, y, colour));
        end;
    end;
    return table.concat(formspec, '');
end;


--- Shows the subtitle menu to the specified player.
-- @param player The username or ObjectRef of the player.
function subtitles.show_menu(player)
    local formspec = subtitles.get_menu_formspec(player);
    if type(player) ~= 'string' then
        player = player:get_player_name();
    end;
    minetest.show_formspec(player, 'subtitles:menu', formspec);
end;


--- Handles receiving fields from the menu formspec.
-- @param player The username or ObjectRef of the player.
-- @param fields A table of fields.
-- @param embedded If this is true, the menu will not be re-shown.
function subtitles.menu_receive_fields(player, fields, embedded)
    local function _reshow()
        if not embedded then
            subtitles.show_menu(player);
        end;
    end;

    local agent = subtitles.get_agent(player);

    if fields.toggle then
        agent:toggle_enabled();
        _reshow();
        return;
    end;

    for __, impl in ipairs(subtitles.SubtitleDisplay.implementations) do
        if fields['display_' .. impl.NAME] then
            agent:set_display_name(impl.NAME);
            _reshow();
            return;
        end;
    end;
end;


-- Unified Inventory integration:
if minetest.get_modpath('unified_inventory') then
	unified_inventory.register_button('subtitles',
	{
		type      = 'image';
		image     = 'subtitles_icon.png';
		tooltip   = S'Subtitles';
		hide_lite = false;
		action    = subtitles.show_menu;
	});
end;


-- i3 integration:
if minetest.get_modpath('i3') then
    i3.new_tab('subtitles',
    {
	    description = S'Subtitles';
	    image       = 'subtitles_icon.png';
	    formspec =
	    function(player, data, fs)
		    fs(subtitles.get_menu_formspec(player, true, {width = 10.25, height = 12}));
	    end;
	    fields =
	    function(player, data, fields)
	        subtitles.menu_receive_fields(player, fields, true);
		    i3.set_fs(player);
	    end;
    });
end;


-- SFInv integration via SFInv Buttons:
if minetest.get_modpath('sfinv_buttons') then
	sfinv_buttons.register_button('subtitles',
	{
		image   = 'subtitles_icon.png';
		tooltip = S'Configure your subtitle preferences';
		title   = S'Subtitles';
		action  = subtitles.show_menu;
	});
end;
