local storage = minetest.get_mod_storage()
local CHUNK_SIZE = 16
local HUD_LIMIT = 8

local protected_chunks = minetest.deserialize(storage:get_string("chunks")) or {}
local player_huds = {}

local is_mineclonia = minetest.get_modpath("mcl_core") ~= nil


local function save()
	storage:set_string("chunks", minetest.serialize(protected_chunks))
end

local function get_chunk_key(pos)
	local cx = math.floor(pos.x / CHUNK_SIZE)
	local cz = math.floor(pos.z / CHUNK_SIZE)
	return cx .. "," .. cz
end

local function get_chunk(pos)
	local key = get_chunk_key(pos)
	return protected_chunks[key], key
end

local function is_admin(name)
	return minetest.check_player_privs(name, {server = true})
end

local function has_access(name, chunk)
	if not chunk then return true end
	if is_admin(name) then return true end
	if chunk.owner == name then return true end
	if chunk.members[name] then return true end
	return false
end


minetest.register_chatcommand("cprotect", {
	description = "Protect current chunk",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local key = get_chunk_key(player:get_pos())
		if protected_chunks[key] then
			return false, "Chunk already protected."
		end

		protected_chunks[key] = {
			owner = name,
			members = {}
		}

		save()
		return true, "Chunk protected."
	end
})

minetest.register_chatcommand("cadd", {
	params = "<player>",
	description = "Add player to this chunk",
	func = function(name, param)
		if param == "" then return false, "Usage: /cadd <player>" end
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if not has_access(name, chunk) then return false, "No permission." end

		chunk.members[param] = true
		save()
		return true, param .. " added."
	end
})

minetest.register_chatcommand("rchunk", {
	params = "<player>",
	description = "Remove player from this chunk",
	func = function(name, param)
		if param == "" then return false, "Usage: /rchunk <player>" end
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if not has_access(name, chunk) then return false, "No permission." end

		chunk.members[param] = nil
		save()
		return true, param .. " removed."
	end
})

minetest.register_chatcommand("cdelete", {
	description = "Delete chunk protection",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk, key = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if not is_admin(name) and chunk.owner ~= name then
			return false, "Only owner or admin."
		end

		protected_chunks[key] = nil
		save()
		return true, "Chunk unprotected."
	end
})


local function show_chunks(player)
	local pos = vector.round(player:get_pos())
	local base_x = math.floor(pos.x / CHUNK_SIZE) * CHUNK_SIZE
	local base_z = math.floor(pos.z / CHUNK_SIZE) * CHUNK_SIZE

	for x = 0, CHUNK_SIZE do
		for z = 0, CHUNK_SIZE do
			minetest.add_particle({
				pos = {x = base_x + x, y = pos.y + 0.1, z = base_z + z},
				expirationtime = 3,
				size = 3,
				texture = "default_mese_crystal.png",
				glow = 5
			})
		end
	end
end

local function register_show_cmd(cmd)
	minetest.register_chatcommand(cmd, {
		description = "Show chunk borders",
		func = function(name)
			local player = minetest.get_player_by_name(name)
			if not player then return end
			show_chunks(player)
		end
	})
end

register_show_cmd("showchunks")
register_show_cmd("schunk")
register_show_cmd("schunks")


local function update_hud(player)
	local name = player:get_player_name()
	local chunk = get_chunk(player:get_pos())

	if player_huds[name] then
		player:hud_remove(player_huds[name])
		player_huds[name] = nil
	end

	if not chunk then return end

	local list = {chunk.owner}
	for m,_ in pairs(chunk.members) do
		if #list >= HUD_LIMIT then
			table.insert(list, "...")
			break
		end
		table.insert(list, m)
	end

	player_huds[name] = player:hud_add({
		hud_elem_type = "text",
		position = {x = 1, y = 1},
		offset = {x = -10, y = -10},
		alignment = {x = -1, y = -1},
		text = "Chunk access:\n" .. table.concat(list, ", "),
		number = 0xFFFFFF
	})
end

local hud_timer = 0
minetest.register_globalstep(function(dtime)
	hud_timer = hud_timer + dtime
	if hud_timer < 1 then return end
	hud_timer = 0

	for _,player in ipairs(minetest.get_connected_players()) do
		update_hud(player)
	end
end)


local function denied(pos, name)
	local chunk = get_chunk(pos)
	return chunk and not has_access(name, chunk)
end

minetest.register_on_dignode(function(pos, oldnode, digger)
	if digger and denied(pos, digger:get_player_name()) then
		minetest.set_node(pos, oldnode)
	end
end)

minetest.register_on_placenode(function(pos, node, placer, oldnode)
	if placer and denied(pos, placer:get_player_name()) then
		minetest.set_node(pos, oldnode)
	end
end)

minetest.register_allow_player_inventory_action(function(player, action, inventory, info)
	if not player then return end
	local loc = inventory:get_location()
	if not loc or not loc.pos then return end

	local chunk = get_chunk(loc.pos)
	if chunk and not has_access(player:get_player_name(), chunk) then
		return 0
	end
end)

-- Global protection (doors, beds, mesecons)
local old_is_protected = minetest.is_protected
minetest.is_protected = function(pos, name)
	if old_is_protected(pos, name) then return true end
	local chunk = get_chunk(pos)
	return chunk and not has_access(name, chunk)
end

-- TNT (Voxel Libre / MTG)
if minetest.get_modpath("tnt") and tnt and tnt.register_on_blast then
	tnt.register_on_blast(function(pos, intensity)
		local chunk = get_chunk(pos)
		if chunk then
			return true
		end
	end)
end

-- Mineclonia explosions
if minetest.get_modpath("mcl_explosions") and mcl_explosions then
	local old_explode = mcl_explosions.explode
	mcl_explosions.explode = function(pos, strength, ...)
		local chunk = get_chunk(pos)
		if chunk then
			return
		end
		return old_explode(pos, strength, ...)
	end
end


if is_mineclonia and minetest.get_modpath("mcl_protection") then
	mcl_protection.register_protection(function(pos, player)
		if not player then return false end
		local chunk = get_chunk(pos)
		return chunk and not has_access(player:get_player_name(), chunk)
	end)
end