local storage = minetest.get_mod_storage()
local CHUNK_SIZE = 16
local HUD_LIMIT = 8
local HUD_UPDATE_INTERVAL = 2
local SAVE_INTERVAL = 30

local protected_chunks = minetest.deserialize(storage:get_string("chunks")) or {}
local player_huds = {}
local player_current_chunk = {}
local chunk_cache = {}
local dirty = false

local is_mineclonia = minetest.get_modpath("mcl_core") ~= nil
local is_voxelibre = minetest.get_modpath("voxelibre") or minetest.get_modpath("mcl_init")

local function save()
	storage:set_string("chunks", minetest.serialize(protected_chunks))
	dirty = false
end

local save_timer = 0
local function mark_dirty()
	dirty = true
end

local function get_chunk_key(pos)
	local cx = math.floor(pos.x / CHUNK_SIZE)
	local cz = math.floor(pos.z / CHUNK_SIZE)
	return cx .. "," .. cz
end

local function get_chunk(pos)
	local key = get_chunk_key(pos)
	if chunk_cache[key] ~= nil then
		return chunk_cache[key], key
	end
	local chunk = protected_chunks[key]
	chunk_cache[key] = chunk
	return chunk, key
end

local function clear_chunk_cache(key)
	chunk_cache[key] = nil
end

local function is_admin(name)
	return minetest.check_player_privs(name, {server = true})
end

local function has_access(name, chunk)
	if not chunk then return true end
	if is_admin(name) then return true end
	if chunk.owner == name then return true end
	if chunk.members[name] then return true end
	return false
end

local function count_player_chunks(name)
	local count = 0
	for _, chunk in pairs(protected_chunks) do
		if chunk.owner == name then
			count = count + 1
		end
	end
	return count
end

local function get_chunk_limit(name)
	if is_admin(name) then return math.huge end
	return tonumber(minetest.settings:get("chunk_protection_limit")) or 10
end

minetest.register_chatcommand("chelp", {
	description = "Show all chunk protection commands",
	func = function(name)
		local is_admin_user = is_admin(name)
		local help_text = "=== Chunk Protection Commands ===\n\n" ..
			"Basic Commands:\n" ..
			"/cprotect - Protect the current chunk\n" ..
			"/cdelete - Delete protection of current chunk\n" ..
			"/cinfo - Show info about current chunk\n" ..
			"/clist - List all your protected chunks\n\n" ..
			"Member Management:\n" ..
			"/cadd <player> - Give player access to your chunk\n" ..
			"/cremove <player> - Remove player from your chunk\n\n" ..
			"Advanced:\n" ..
			"/ctransfer <player> - Transfer chunk ownership\n" ..
			"/cpublic - Toggle public access for this chunk\n" ..
			"/cshow - Show chunk borders with particles"

		if is_admin_user then
			help_text = help_text .. "\n\nAdmin Commands:\n" ..
				"/cadminlist - List all protected chunks (admin only)"
		end

		help_text = help_text .. "\n\nExamples:\n" ..
			"/cprotect - Protect where you stand\n" ..
			"/cadd Steve - Give Steve access\n" ..
			"/cremove Steve - Remove Steve's access\n" ..
			"/cpublic - Make chunk public/private"

		local formspec = "size[12,10]" ..
			"bgcolor[#00000099;true]" ..
			"box[0.3,0.3;11.4,9.4;#333333]" ..
			"textarea[0.5,0.5;11.5,9.5;;;" .. minetest.formspec_escape(help_text) .. "]" ..
			"button_exit[4.5,9.2;3,0.8;close;Close]"

		minetest.show_formspec(name, "chunk_protection:help", formspec)
		return true
	end
})

minetest.register_chatcommand("cprotect", {
	description = "Protect current chunk",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local pos = player:get_pos()
		local key = get_chunk_key(pos)
		
		if protected_chunks[key] then
			return false, "Chunk already protected."
		end

		local count = count_player_chunks(name)
		local limit = get_chunk_limit(name)
		if count >= limit then
			return false, string.format("Chunk limit reached (%d/%d). Use /cdelete to free chunks.", count, limit)
		end

		protected_chunks[key] = {
			owner = name,
			members = {},
			created = os.time(),
			last_visit = os.time()
		}

		clear_chunk_cache(key)
		mark_dirty()
		return true, string.format("Chunk protected. (%d/%d chunks used)", count + 1, limit)
	end
})

minetest.register_chatcommand("cadd", {
	params = "<player>",
	description = "Add player to this chunk",
	func = function(name, param)
		if param == "" then return false, "Usage: /cadd <player>" end
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk, key = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if not has_access(name, chunk) then return false, "No permission." end

		if chunk.members[param] then
			return false, param .. " already has access."
		end

		chunk.members[param] = true
		clear_chunk_cache(key)
		mark_dirty()
		
		local target = minetest.get_player_by_name(param)
		if target then
			minetest.chat_send_player(param, string.format("%s granted you access to their chunk.", name))
		end
		
		return true, param .. " added."
	end
})

minetest.register_chatcommand("cremove", {
	params = "<player>",
	description = "Remove player from this chunk",
	func = function(name, param)
		if param == "" then return false, "Usage: /cremove <player>" end
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk, key = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if not has_access(name, chunk) then return false, "No permission." end

		if not chunk.members[param] then
			return false, param .. " doesn't have access."
		end

		chunk.members[param] = nil
		clear_chunk_cache(key)
		mark_dirty()
		
		local target = minetest.get_player_by_name(param)
		if target then
			minetest.chat_send_player(param, string.format("%s revoked your access to their chunk.", name))
		end
		
		return true, param .. " removed."
	end
})

minetest.register_chatcommand("cdelete", {
	description = "Delete chunk protection",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk, key = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if not is_admin(name) and chunk.owner ~= name then
			return false, "Only owner or admin."
		end

		protected_chunks[key] = nil
		clear_chunk_cache(key)
		mark_dirty()
		
		local count = count_player_chunks(name)
		local limit = get_chunk_limit(name)
		return true, string.format("Chunk unprotected. (%d/%d chunks used)", count, limit)
	end
})

minetest.register_chatcommand("cinfo", {
	description = "Show info about current chunk",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk = get_chunk(player:get_pos())
		if not chunk then
			return true, "This chunk is not protected."
		end

		local members = {}
		for m,_ in pairs(chunk.members) do
			table.insert(members, m)
		end

		local info = string.format(
			"Chunk Owner: %s\nMembers: %s\nCreated: %s",
			chunk.owner,
			#members > 0 and table.concat(members, ", ") or "None",
			os.date("%Y-%m-%d", chunk.created or 0)
		)
		
		return true, info
	end
})

minetest.register_chatcommand("clist", {
	description = "List all your protected chunks",
	func = function(name)
		local chunks = {}
		for key, chunk in pairs(protected_chunks) do
			if chunk.owner == name then
				table.insert(chunks, key)
			end
		end

		if #chunks == 0 then
			return true, "You don't own any protected chunks."
		end

		local limit = get_chunk_limit(name)
		return true, string.format("Your chunks (%d/%d):\n%s", #chunks, limit, table.concat(chunks, "\n"))
	end
})

minetest.register_chatcommand("ctransfer", {
	params = "<player>",
	description = "Transfer chunk ownership to another player",
	func = function(name, param)
		if param == "" then return false, "Usage: /ctransfer <player>" end
		
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk, key = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if chunk.owner ~= name and not is_admin(name) then
			return false, "Only the owner can transfer."
		end

		local target_count = count_player_chunks(param)
		local target_limit = get_chunk_limit(param)
		if target_count >= target_limit then
			return false, string.format("%s has reached their chunk limit (%d/%d).", param, target_count, target_limit)
		end

		chunk.owner = param
		chunk.members[name] = true
		clear_chunk_cache(key)
		mark_dirty()
		
		local target = minetest.get_player_by_name(param)
		if target then
			minetest.chat_send_player(param, string.format("%s transferred a chunk to you.", name))
		end
		
		return true, string.format("Chunk transferred to %s.", param)
	end
})

minetest.register_chatcommand("cpublic", {
	description = "Toggle public access for this chunk",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end

		local chunk, key = get_chunk(player:get_pos())
		if not chunk then return false, "Chunk not protected." end
		if chunk.owner ~= name and not is_admin(name) then
			return false, "Only owner can change public access."
		end

		chunk.public = not chunk.public
		clear_chunk_cache(key)
		mark_dirty()
		
		return true, chunk.public and "Chunk is now public." or "Chunk is now private."
	end
})

minetest.register_chatcommand("cadminlist", {
	description = "List all protected chunks (admin only)",
	privs = {server = true},
	func = function(name)
		local count = 0
		local owners = {}
		
		for _, chunk in pairs(protected_chunks) do
			count = count + 1
			owners[chunk.owner] = (owners[chunk.owner] or 0) + 1
		end

		local result = string.format("Total protected chunks: %d\n\nChunks per player:\n", count)
		for owner, num in pairs(owners) do
			result = result .. string.format("%s: %d\n", owner, num)
		end
		
		return true, result
	end
})

local function show_chunks(player)
	local pos = vector.round(player:get_pos())
	local base_x = math.floor(pos.x / CHUNK_SIZE) * CHUNK_SIZE
	local base_z = math.floor(pos.z / CHUNK_SIZE) * CHUNK_SIZE

	for x = 0, CHUNK_SIZE, 4 do
		for z = 0, CHUNK_SIZE, 4 do
			minetest.add_particle({
				pos = {x = base_x + x, y = pos.y + 0.1, z = base_z + z},
				expirationtime = 3,
				size = 3,
				texture = "default_mese_crystal.png",
				glow = 5
			})
		end
	end
end

minetest.register_chatcommand("cshow", {
	description = "Show chunk borders",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		show_chunks(player)
		return true, "Chunk borders shown."
	end
})

local function update_hud(player)
	local name = player:get_player_name()
	local pos = player:get_pos()
	local key = get_chunk_key(pos)
	
	if player_current_chunk[name] == key then
		return
	end
	
	player_current_chunk[name] = key
	
	if player_huds[name] then
		player:hud_remove(player_huds[name])
		player_huds[name] = nil
	end

	local chunk = get_chunk(pos)
	if not chunk then return end

	if chunk.last_visit then
		chunk.last_visit = os.time()
	end

	local access_text = chunk.public and "Public" or ""
	if chunk.owner == name then
		access_text = "Owner" .. (chunk.public and " (Public)" or "")
	elseif chunk.members[name] then
		access_text = "Member" .. (chunk.public and " (Public)" or "")
	elseif chunk.public then
		access_text = "Public"
	end

	local list = {chunk.owner}
	for m,_ in pairs(chunk.members) do
		if #list >= HUD_LIMIT then
			table.insert(list, "...")
			break
		end
		table.insert(list, m)
	end

	player_huds[name] = player:hud_add({
		hud_elem_type = "text",
		position = {x = 1, y = 1},
		offset = {x = -10, y = -10},
		alignment = {x = -1, y = -1},
		text = string.format("Chunk: %s\nAccess: %s", table.concat(list, ", "), access_text),
		number = 0xFFFFFF
	})
end

local hud_timer = 0
minetest.register_globalstep(function(dtime)
	hud_timer = hud_timer + dtime
	save_timer = save_timer + dtime
	
	if hud_timer >= HUD_UPDATE_INTERVAL then
		hud_timer = 0
		for _, player in ipairs(minetest.get_connected_players()) do
			update_hud(player)
		end
	end
	
	if save_timer >= SAVE_INTERVAL and dirty then
		save_timer = 0
		save()
	end
end)

local function denied(pos, name)
	local chunk = get_chunk(pos)
	if not chunk then return false end
	if chunk.public then return false end
	return not has_access(name, chunk)
end

minetest.register_on_dignode(function(pos, oldnode, digger)
	if digger and denied(pos, digger:get_player_name()) then
		minetest.set_node(pos, oldnode)
		minetest.chat_send_player(digger:get_player_name(), minetest.colorize("#FF5555", "This chunk is protected!"))
		return true
	end
end)

minetest.register_on_placenode(function(pos, node, placer, oldnode)
	if placer and denied(pos, placer:get_player_name()) then
		minetest.set_node(pos, oldnode)
		minetest.chat_send_player(placer:get_player_name(), minetest.colorize("#FF5555", "This chunk is protected!"))
		return true
	end
end)

minetest.register_allow_player_inventory_action(function(player, action, inventory, info)
	if not player then return end
	local loc = inventory:get_location()
	if not loc or not loc.pos then return end

	local chunk = get_chunk(loc.pos)
	if chunk and not chunk.public and not has_access(player:get_player_name(), chunk) then
		return 0
	end
end)

local old_is_protected = minetest.is_protected
minetest.is_protected = function(pos, name)
	if old_is_protected(pos, name) then return true end
	return denied(pos, name)
end

if minetest.get_modpath("tnt") and tnt and tnt.register_on_blast then
	tnt.register_on_blast(function(pos, intensity)
		local chunk = get_chunk(pos)
		if chunk then
			return true
		end
	end)
end

if is_voxelibre and minetest.get_modpath("mcl_explosions") and mcl_explosions then
	local old_explode = mcl_explosions.explode
	mcl_explosions.explode = function(pos, strength, ...)
		local chunk = get_chunk(pos)
		if chunk then
			return
		end
		return old_explode(pos, strength, ...)
	end
end

if is_mineclonia and minetest.registered_entities["mcl_tnt:tnt"] then
	local old_on_blast = minetest.registered_entities["mcl_tnt:tnt"].on_blast
	if old_on_blast then
		minetest.registered_entities["mcl_tnt:tnt"].on_blast = function(self, damage)
			local pos = self.object:get_pos()
			local chunk = get_chunk(pos)
			if chunk then
				return
			end
			return old_on_blast(self, damage)
		end
	end
end

if is_mineclonia and minetest.get_modpath("mcl_protection") then
	mcl_protection.register_protection(function(pos, player)
		if not player then return false end
		return denied(pos, player:get_player_name())
	end)
end

minetest.register_on_leaveplayer(function(player)
	local name = player:get_player_name()
	player_huds[name] = nil
	player_current_chunk[name] = nil
end)

minetest.register_on_shutdown(function()
	if dirty then
		save()
	end
end)

minetest.log("action", "[Chunk Protection] Loaded")