local DEAD_KEY = "hardcore_luanti_dead"
local CLEAR_INTERVAL = 600

local function detect_game()
    if minetest.get_modpath("mcl_core") then
        return "mineclonia"
    elseif minetest.get_modpath("voxelibre") or minetest.get_modpath("mcl_init") then
        return "voxelibre"
    else
        return "minetest_game"
    end
end

local GAME_TYPE = detect_game()

local function is_dead(player)
    return player:get_meta():get_string(DEAD_KEY) == "true"
end

local function get_max_hp(player)
    local props = player:get_properties()
    return props.hp_max or 20
end

local function enforce_privs(name)
    local current_privs = minetest.get_player_privs(name)
    
    local restricted_privs = {
        interact = nil,
        shout = nil,
        bring = nil,
        give = nil,
        settime = nil,
        privs = nil,
        basic_privs = nil,
    }
    
    for priv, _ in pairs(restricted_privs) do
        current_privs[priv] = nil
    end
    
    current_privs.fly = true
    current_privs.fast = true
    current_privs.teleport = true
    current_privs.noclip = true
    
    minetest.set_player_privs(name, current_privs)
end

local function clear_inventory(player)
    local inv = player:get_inventory()
    
    inv:set_list("main", {})
    inv:set_list("craft", {})
    
    if GAME_TYPE == "voxelibre" or GAME_TYPE == "mineclonia" then
        inv:set_list("armor", {})
        inv:set_list("offhand", {})
        inv:set_list("craftpreview", {})
    else
        if inv:get_list("craftpreview") then
            inv:set_list("craftpreview", {})
        end
    end
end

local function apply_dead_state(player)
    local meta = player:get_meta()
    meta:set_string(DEAD_KEY, "true")
    
    local name = player:get_player_name()
    enforce_privs(name)
    
    player:set_armor_groups({ immortal = 1, fleshy = 0 })
    
    player:set_properties({
        visual_size = {x = 0.8, y = 0.8},
        makes_footstep_sound = false,
    })
    
    player:set_nametag_attributes({
        color = {a = 128, r = 128, g = 128, b = 128},
        text = name .. " (DEAD)"
    })
end

minetest.register_on_dieplayer(function(player)
    minetest.after(0.1, function()
        if not player or not player:is_player() then return end
        
        player:set_hp(get_max_hp(player))
        apply_dead_state(player)
        clear_inventory(player)
        
        minetest.chat_send_player(player:get_player_name(), 
            minetest.colorize("#FF0000", "You have died! You are now in spectator mode."))
    end)
    return true
end)

minetest.register_on_joinplayer(function(player)
    if is_dead(player) then
        minetest.after(0.1, function()
            if not player or not player:is_player() then return end
            apply_dead_state(player)
        end)
    end
end)

local timer = 0
minetest.register_globalstep(function(dtime)
    timer = timer + dtime
    
    for _, player in ipairs(minetest.get_connected_players()) do
        if is_dead(player) then
            local name = player:get_player_name()
            
            local groups = player:get_armor_groups()
            if not groups.immortal or groups.immortal ~= 1 then
                player:set_armor_groups({ immortal = 1, fleshy = 0 })
            end
            
            local pos = player:get_pos()
            local eye_pos = {x = pos.x, y = pos.y + player:get_properties().eye_height, z = pos.z}
            local node = minetest.get_node(eye_pos)
            local node_def = minetest.registered_nodes[node.name]
            
            if node_def and not node_def.walkable == false and node.name ~= "air" and 
               node.name ~= "ignore" and node_def.drawtype ~= "airlike" then
                player:hud_set_flags({
                    crosshair = false,
                    hotbar = false,
                    healthbar = false,
                    breathbar = false,
                    minimap = false,
                    basic_debug = false,
                })
                player:override_day_night_ratio(0)
            else
                player:hud_set_flags({
                    crosshair = true,
                    hotbar = true,
                    healthbar = true,
                    breathbar = true,
                    minimap = true,
                    basic_debug = true,
                })
                player:override_day_night_ratio(nil)
            end
            
            if timer >= CLEAR_INTERVAL then
                clear_inventory(player)
            end
            
            local inv = player:get_inventory()
            if not inv:is_empty("main") or not inv:is_empty("craft") then
                clear_inventory(player)
            end
        end
    end
    
    if timer >= CLEAR_INTERVAL then
        timer = 0
    end
end)

minetest.register_on_chat_message(function(name, message)
    local player = minetest.get_player_by_name(name)
    if player and is_dead(player) then
        minetest.chat_send_player(name, 
            minetest.colorize("#FF0000", "You cannot chat while dead!"))
        return true
    end
end)

minetest.register_on_player_hpchange(function(player, hp_change, reason)
    if is_dead(player) then
        return 0
    end
    
    if reason and reason.object and reason.object:is_player() then
        if is_dead(reason.object) then
            return 0
        end
    end
    
    return hp_change
end, true)

minetest.register_on_punchplayer(function(player, hitter)
    if hitter and hitter:is_player() and is_dead(hitter) then
        return true
    end
end)

minetest.register_on_item_pickup(function(itemstack, picker)
    if picker and is_dead(picker) then
        return itemstack
    end
end)

minetest.register_on_placenode(function(pos, newnode, placer)
    if placer and is_dead(placer) then
        minetest.remove_node(pos)
        return true
    end
end)

minetest.register_on_dignode(function(pos, oldnode, digger)
    if digger and is_dead(digger) then
        minetest.set_node(pos, oldnode)
        return true
    end
end)

minetest.register_allow_player_inventory_action(function(player, action)
    if is_dead(player) then
        return 0
    end
end)

minetest.register_on_priv_grant(function(name, granter, priv)
    local player = minetest.get_player_by_name(name)
    if player and is_dead(player) then
        minetest.after(0, function()
            enforce_privs(name)
        end)
    end
end)

minetest.log("action", "[Hardcore Luanti] Loaded for game type: " .. GAME_TYPE)