-- Hardcore Luanti (Performance-Optimized + Persistent Privileges)
local DEAD_KEY = "hardcore_luanti_dead"
local PERMANENT_RESTRICTION_KEY = "hardcore_luanti_restricted"
local CLEAR_INTERVAL = 600
local CHECK_INTERVAL = 5 -- Reduziert von jedem Frame auf alle 5 Sekunden

local function detect_game()
    if minetest.get_modpath("mcl_core") then
        return "mineclonia"
    elseif minetest.get_modpath("voxelibre") or minetest.get_modpath("mcl_init") then
        return "voxelibre"
    else
        return "minetest_game"
    end
end

local GAME_TYPE = detect_game()

-- Cache für tote Spieler (vermeidet Meta-Zugriffe)
local dead_players = {}
local restricted_players = {}

local function is_dead(player)
    local name = player:get_player_name()
    if dead_players[name] ~= nil then
        return dead_players[name]
    end
    
    local is_dead_state = player:get_meta():get_string(DEAD_KEY) == "true"
    dead_players[name] = is_dead_state
    return is_dead_state
end

local function mark_permanently_restricted(name)
    local player = minetest.get_player_by_name(name)
    if player then
        player:get_meta():set_string(PERMANENT_RESTRICTION_KEY, "true")
    end
    restricted_players[name] = true
end

local function is_permanently_restricted(name)
    if restricted_players[name] ~= nil then
        return restricted_players[name]
    end
    
    local player = minetest.get_player_by_name(name)
    if player then
        local is_restricted = player:get_meta():get_string(PERMANENT_RESTRICTION_KEY) == "true"
        restricted_players[name] = is_restricted
        return is_restricted
    end
    return false
end

local function get_max_hp(player)
    return player:get_properties().hp_max or 20
end

local function enforce_privs(name)
    local current_privs = minetest.get_player_privs(name)
    
    -- Entferne kritische Privilegien
    current_privs.interact = nil
    current_privs.shout = nil
    current_privs.bring = nil
    current_privs.give = nil
    current_privs.settime = nil
    current_privs.privs = nil
    current_privs.basic_privs = nil
    
    -- Gebe Spectator-Privilegien
    current_privs.fly = true
    current_privs.fast = true
    current_privs.teleport = true
    current_privs.noclip = true
    
    minetest.set_player_privs(name, current_privs)
end

local function clear_inventory(player)
    local inv = player:get_inventory()
    
    inv:set_list("main", {})
    inv:set_list("craft", {})
    
    if GAME_TYPE == "voxelibre" or GAME_TYPE == "mineclonia" then
        inv:set_list("armor", {})
        inv:set_list("offhand", {})
        inv:set_list("craftpreview", {})
    elseif inv:get_list("craftpreview") then
        inv:set_list("craftpreview", {})
    end
end

local function apply_dead_state(player)
    local meta = player:get_meta()
    meta:set_string(DEAD_KEY, "true")
    
    local name = player:get_player_name()
    dead_players[name] = true
    
    -- Markiere als permanent eingeschränkt
    mark_permanently_restricted(name)
    
    enforce_privs(name)
    
    player:set_armor_groups({ immortal = 1, fleshy = 0 })
    
    player:set_properties({
        visual_size = {x = 0.8, y = 0.8},
        makes_footstep_sound = false,
    })
    
    player:set_nametag_attributes({
        color = {a = 128, r = 128, g = 128, b = 128},
        text = name .. " (DEAD)"
    })
end

minetest.register_on_dieplayer(function(player)
    minetest.after(0.1, function()
        if not player or not player:is_player() then return end
        
        player:set_hp(get_max_hp(player))
        apply_dead_state(player)
        clear_inventory(player)
        
        minetest.chat_send_player(player:get_player_name(), 
            minetest.colorize("#FF0000", "You have died! You are now in spectator mode forever."))
    end)
    return true
end)

minetest.register_on_joinplayer(function(player)
    local name = player:get_player_name()
    
    -- Lade Restricted-Status in Cache
    restricted_players[name] = player:get_meta():get_string(PERMANENT_RESTRICTION_KEY) == "true"
    
    -- Wenn permanent eingeschränkt, erzwinge Privilegien auch ohne Dead-Status
    if is_permanently_restricted(name) then
        minetest.after(0.1, function()
            enforce_privs(name)
        end)
    end
    
    if is_dead(player) then
        minetest.after(0.1, function()
            if not player or not player:is_player() then return end
            apply_dead_state(player)
        end)
    end
end)

-- Performance-optimierter Globalstep
local check_timer = 0
local clear_timer = 0
local cached_dead_list = {}
local cache_refresh_counter = 0

minetest.register_globalstep(function(dtime)
    check_timer = check_timer + dtime
    clear_timer = clear_timer + dtime
    
    -- Nur alle CHECK_INTERVAL Sekunden prüfen
    if check_timer < CHECK_INTERVAL then return end
    check_timer = 0
    
    -- Refresh Cache alle 3 Durchläufe
    cache_refresh_counter = cache_refresh_counter + 1
    if cache_refresh_counter >= 3 then
        cached_dead_list = {}
        cache_refresh_counter = 0
    end
    
    -- Baue Liste toter Spieler (nur wenn Cache leer)
    if #cached_dead_list == 0 then
        for _, player in ipairs(minetest.get_connected_players()) do
            if is_dead(player) then
                table.insert(cached_dead_list, player)
            end
        end
    end
    
    -- Verarbeite nur tote Spieler
    for _, player in ipairs(cached_dead_list) do
        if player:is_player() then
            -- Armor-Gruppen Check (nur wenn nötig)
            local groups = player:get_armor_groups()
            if groups.immortal ~= 1 then
                player:set_armor_groups({ immortal = 1, fleshy = 0 })
            end
            
            -- Node-basierte HUD-Anpassung (optimiert)
            local pos = player:get_pos()
            local eye_offset = player:get_properties().eye_height or 1.625
            local node = minetest.get_node({x = pos.x, y = pos.y + eye_offset, z = pos.z})
            
            if node.name ~= "air" and node.name ~= "ignore" then
                local node_def = minetest.registered_nodes[node.name]
                if node_def and node_def.walkable and node_def.drawtype ~= "airlike" then
                    player:hud_set_flags({
                        crosshair = false,
                        hotbar = false,
                        healthbar = false,
                        breathbar = false,
                        minimap = false,
                        basic_debug = false,
                    })
                    player:override_day_night_ratio(0)
                else
                    player:hud_set_flags({
                        crosshair = true,
                        hotbar = true,
                        healthbar = true,
                        breathbar = true,
                        minimap = true,
                        basic_debug = true,
                    })
                    player:override_day_night_ratio(nil)
                end
            end
            
            -- Inventory-Check (nur periodisch)
            if clear_timer >= CLEAR_INTERVAL then
                clear_inventory(player)
            else
                local inv = player:get_inventory()
                if not inv:is_empty("main") or not inv:is_empty("craft") then
                    clear_inventory(player)
                end
            end
        end
    end
    
    if clear_timer >= CLEAR_INTERVAL then
        clear_timer = 0
    end
end)

-- Optimierte Callbacks mit frühem Return
minetest.register_on_chat_message(function(name, message)
    if dead_players[name] then
        minetest.chat_send_player(name, 
            minetest.colorize("#FF0000", "You cannot chat while dead!"))
        return true
    end
end)

minetest.register_on_player_hpchange(function(player, hp_change, reason)
    if dead_players[player:get_player_name()] then
        return 0
    end
    
    if reason and reason.object and reason.object:is_player() then
        if dead_players[reason.object:get_player_name()] then
            return 0
        end
    end
    
    return hp_change
end, true)

minetest.register_on_punchplayer(function(player, hitter)
    if hitter and hitter:is_player() and dead_players[hitter:get_player_name()] then
        return true
    end
end)

minetest.register_on_item_pickup(function(itemstack, picker)
    if picker and dead_players[picker:get_player_name()] then
        return itemstack
    end
end)

minetest.register_on_placenode(function(pos, newnode, placer)
    if placer and dead_players[placer:get_player_name()] then
        minetest.remove_node(pos)
        return true
    end
end)

minetest.register_on_dignode(function(pos, oldnode, digger)
    if digger and dead_players[digger:get_player_name()] then
        minetest.set_node(pos, oldnode)
        return true
    end
end)

minetest.register_allow_player_inventory_action(function(player)
    if dead_players[player:get_player_name()] then
        return 0
    end
end)

minetest.register_on_priv_grant(function(name, granter, priv)
    if restricted_players[name] then
        minetest.after(0, function()
            enforce_privs(name)
        end)
    end
end)

minetest.register_on_priv_revoke(function(name, revoker, priv)
    if restricted_players[name] then
        minetest.after(0, function()
            enforce_privs(name)
        end)
    end
end)

-- Cleanup bei Disconnect
minetest.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    dead_players[name] = nil
    -- restricted_players bleibt bestehen für permanente Einschränkung
end)

minetest.log("action", "[Hardcore Luanti] Loaded (Performance Mode) for: " .. GAME_TYPE)