local favorites = {}

local function load_favorites(player_name)
	local meta = minetest.get_player_by_name(player_name)
	if not meta then return {} end
	local data = meta:get_meta():get_string("inventory_favorites")
	return minetest.deserialize(data) or {}
end

local function save_favorites(player_name, favs)
	local player = minetest.get_player_by_name(player_name)
	if not player then return end
	player:get_meta():set_string("inventory_favorites", minetest.serialize(favs))
end

local function get_item_sort_key(itemstack)
	if itemstack:is_empty() then return "zzz_empty" end
	local name = itemstack:get_name()
	local def = minetest.registered_items[name]
	local category = "other"
	
	if def then
		if def.groups.soil then category = "a_farming"
		elseif def.groups.tree then category = "b_wood"
		elseif def.groups.stone then category = "c_stone"
		elseif def.groups.ore then category = "d_ore"
		elseif name:find("ingot") then category = "e_ingots"
		elseif def.type == "tool" then category = "f_tools"
		elseif def.groups.armor then category = "g_armor"
		elseif def.groups.food then category = "h_food"
		end
	end
	
	return category .. "_" .. name
end

local function sort_inventory(player, listname)
	local inv = player:get_inventory()
	if not inv then return false end
	
	local items = {}
	local size = inv:get_size(listname)
	
	for i = 1, size do
		local stack = inv:get_stack(listname, i)
		if not stack:is_empty() then
			table.insert(items, {
				stack = stack,
				sort_key = get_item_sort_key(stack)
			})
		end
	end
	
	table.sort(items, function(a, b)
		return a.sort_key < b.sort_key
	end)
	
	inv:set_list(listname, {})
	
	for i, item in ipairs(items) do
		inv:set_stack(listname, i, item.stack)
	end
	
	return true
end

local function stack_all(player, listname)
	local inv = player:get_inventory()
	if not inv then return false end
	
	local size = inv:get_size(listname)
	local stacks = {}
	
	for i = 1, size do
		local stack = inv:get_stack(listname, i)
		if not stack:is_empty() then
			local name = stack:get_name()
			if not stacks[name] then
				stacks[name] = {}
			end
			table.insert(stacks[name], {index = i, stack = stack})
		end
	end
	
	for name, stack_list in pairs(stacks) do
		if #stack_list > 1 then
			local max_stack = minetest.registered_items[name].stack_max or 99
			local total = 0
			
			for _, s in ipairs(stack_list) do
				total = total + s.stack:get_count()
			end
			
			for _, s in ipairs(stack_list) do
				inv:set_stack(listname, s.index, ItemStack(""))
			end
			
			local slots_needed = math.ceil(total / max_stack)
			local current_slot = 1
			
			for i = 1, size do
				if inv:get_stack(listname, i):is_empty() and slots_needed > 0 then
					local count = math.min(total, max_stack)
					inv:set_stack(listname, i, ItemStack(name .. " " .. count))
					total = total - count
					slots_needed = slots_needed - 1
					if total <= 0 then break end
				end
			end
		end
	end
	
	return true
end

local function quick_move_to_chest(player, chest_pos)
	local player_inv = player:get_inventory()
	local chest_meta = minetest.get_meta(chest_pos)
	local chest_inv = chest_meta:get_inventory()
	
	if not chest_inv then return false end
	
	local moved = 0
	
	for i = 1, player_inv:get_size("main") do
		local stack = player_inv:get_stack("main", i)
		if not stack:is_empty() then
			local leftover = chest_inv:add_item("main", stack)
			player_inv:set_stack("main", i, leftover)
			if leftover:is_empty() or leftover:get_count() < stack:get_count() then
				moved = moved + 1
			end
		end
	end
	
	return moved > 0, moved
end

local function quick_move_from_chest(player, chest_pos)
	local player_inv = player:get_inventory()
	local chest_meta = minetest.get_meta(chest_pos)
	local chest_inv = chest_meta:get_inventory()
	
	if not chest_inv then return false end
	
	local moved = 0
	
	for i = 1, chest_inv:get_size("main") do
		local stack = chest_inv:get_stack("main", i)
		if not stack:is_empty() then
			local leftover = player_inv:add_item("main", stack)
			chest_inv:set_stack("main", i, leftover)
			if leftover:is_empty() or leftover:get_count() < stack:get_count() then
				moved = moved + 1
			end
		end
	end
	
	return moved > 0, moved
end

minetest.register_chatcommand("sort", {
	description = "Sort your inventory",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		if sort_inventory(player, "main") then
			return true, "Inventory sorted!"
		else
			return false, "Failed to sort inventory."
		end
	end
})

minetest.register_chatcommand("stack", {
	description = "Stack all items in inventory",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		if stack_all(player, "main") then
			return true, "Items stacked!"
		else
			return false, "Failed to stack items."
		end
	end
})

minetest.register_chatcommand("sortstack", {
	description = "Sort and stack inventory",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		stack_all(player, "main")
		sort_inventory(player, "main")
		return true, "Inventory sorted and stacked!"
	end
})

local function show_inventory_menu(player)
	local name = player:get_player_name()
	
	local formspec = "size[10,8]" ..
		"bgcolor[#00000099;true]" ..
		"label[0.5,0.5;Manage Inventory Manager]" ..
		"button[0.5,1.5;3,0.8;sort;Sort Inventory]" ..
		"button[3.7,1.5;3,0.8;stack;Stack All Items]" ..
		"button[6.9,1.5;3,0.8;sortstack;Sort & Stack]" ..
		"label[0.5,2.8;Quick Actions:]" ..
		"button[0.5,3.3;4.5,0.8;deposit;Deposit All to Chest]" ..
		"button[5.2,3.3;4.5,0.8;withdraw;Withdraw All from Chest]" ..
		"label[0.5,4.5;Instructions:]" ..
		"label[0.5,5.0;Sort: Organizes items by category]" ..
		"label[0.5,5.4;Stack: Combines duplicate items]" ..
		"label[0.5,5.8;Sort & Stack: Does both]" ..
		"label[0.5,6.2;Deposit/Withdraw: Look at a chest and click]" ..
		"button_exit[3.5,7.2;3,0.8;close;Close]"
	
	minetest.show_formspec(name, "manage_inventory:menu", formspec)
end

minetest.register_chatcommand("inv", {
	description = "Open Manage Inventory Manager",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		show_inventory_menu(player)
		return true
	end
})

minetest.register_on_player_receive_fields(function(player, formname, fields)
	if formname ~= "manage_inventory:menu" then return end
	
	local name = player:get_player_name()
	
	if fields.sort then
		sort_inventory(player, "main")
		minetest.chat_send_player(name, "Inventory sorted!")
		show_inventory_menu(player)
		
	elseif fields.stack then
		stack_all(player, "main")
		minetest.chat_send_player(name, "Items stacked!")
		show_inventory_menu(player)
		
	elseif fields.sortstack then
		stack_all(player, "main")
		sort_inventory(player, "main")
		minetest.chat_send_player(name, "Inventory sorted and stacked!")
		show_inventory_menu(player)
		
	elseif fields.deposit or fields.withdraw then
		local eye_pos = player:get_pos()
		eye_pos.y = eye_pos.y + player:get_properties().eye_height
		local look_dir = player:get_look_dir()
		local target_pos = vector.add(eye_pos, vector.multiply(look_dir, 5))
		
		local nodes_in_range = minetest.find_nodes_in_area(
			vector.subtract(target_pos, {x=3, y=3, z=3}),
			vector.add(target_pos, {x=3, y=3, z=3}),
			{"default:chest", "default:chest_locked", "mcl_chests:chest", 
			 "mcl_chests:chest_small", "mcl_barrels:barrel_closed"}
		)
		
		if #nodes_in_range == 0 then
			minetest.chat_send_player(name, "No chest found nearby! Look at a chest.")
			show_inventory_menu(player)
			return
		end
		
		local closest_chest = nodes_in_range[1]
		
		if fields.deposit then
			local success, count = quick_move_to_chest(player, closest_chest)
			if success then
				minetest.chat_send_player(name, string.format("Deposited %d stack(s)!", count))
			else
				minetest.chat_send_player(name, "Failed to deposit items.")
			end
		else
			local success, count = quick_move_from_chest(player, closest_chest)
			if success then
				minetest.chat_send_player(name, string.format("Withdrew %d stack(s)!", count))
			else
				minetest.chat_send_player(name, "Failed to withdraw items.")
			end
		end
		
		show_inventory_menu(player)
	end
end)

minetest.register_on_joinplayer(function(player)
	minetest.after(2, function()
		if not player or not player:is_player() then return end
		local name = player:get_player_name()
		minetest.chat_send_player(name, 
			minetest.colorize("#00FF00", "[Manage Inventory] Type /inv for inventory management!"))
	end)
end)