local favorites_cache = {}

local function load_favorites(player_name)
	if favorites_cache[player_name] then
		return favorites_cache[player_name]
	end
	
	local player = minetest.get_player_by_name(player_name)
	if not player then return {} end
	
	local data = player:get_meta():get_string("inventory_favorites")
	local favs = minetest.deserialize(data) or {}
	favorites_cache[player_name] = favs
	return favs
end

local function save_favorites(player_name, favs)
	local player = minetest.get_player_by_name(player_name)
	if not player then return end
	
	favorites_cache[player_name] = favs
	player:get_meta():set_string("inventory_favorites", minetest.serialize(favs))
end

local category_priority = {
	soil = "a_farming",
	tree = "b_wood",
	stone = "c_stone",
	ore = "d_ore",
	food = "h_food",
	armor = "g_armor",
}

local function get_item_sort_key(itemstack)
	if itemstack:is_empty() then return "zzz_empty" end
	
	local name = itemstack:get_name()
	local def = minetest.registered_items[name]
	local category = "other"
	
	if def then
		for group, cat in pairs(category_priority) do
			if def.groups[group] then
				category = cat
				break
			end
		end
		
		if category == "other" then
			if name:find("ingot") then
				category = "e_ingots"
			elseif def.type == "tool" then
				category = "f_tools"
			end
		end
	end
	
	return category .. "_" .. name
end

local function sort_inventory(player, listname)
	local inv = player:get_inventory()
	if not inv then return false end
	
	local size = inv:get_size(listname)
	local items = {}
	
	for i = 1, size do
		local stack = inv:get_stack(listname, i)
		if not stack:is_empty() then
			items[#items + 1] = {
				stack = stack,
				sort_key = get_item_sort_key(stack)
			}
		end
	end
	
	table.sort(items, function(a, b)
		return a.sort_key < b.sort_key
	end)
	
	inv:set_list(listname, {})
	
	for i = 1, #items do
		inv:set_stack(listname, i, items[i].stack)
	end
	
	return true
end

local function stack_all(player, listname)
	local inv = player:get_inventory()
	if not inv then return false end
	
	local size = inv:get_size(listname)
	local item_groups = {}
	
	for i = 1, size do
		local stack = inv:get_stack(listname, i)
		if not stack:is_empty() then
			local name = stack:get_name()
			local count = stack:get_count()
			
			if not item_groups[name] then
				item_groups[name] = 0
			end
			item_groups[name] = item_groups[name] + count
		end
	end
	
	inv:set_list(listname, {})
	
	local slot = 1
	for name, total_count in pairs(item_groups) do
		local max_stack = minetest.registered_items[name].stack_max or 99
		
		while total_count > 0 and slot <= size do
			local count = math.min(total_count, max_stack)
			inv:set_stack(listname, slot, ItemStack(name .. " " .. count))
			total_count = total_count - count
			slot = slot + 1
		end
	end
	
	return true
end

local function quick_move_to_chest(player, chest_pos)
	local player_inv = player:get_inventory()
	local chest_meta = minetest.get_meta(chest_pos)
	local chest_inv = chest_meta:get_inventory()
	
	if not chest_inv then return false, 0 end
	
	local moved = 0
	local size = player_inv:get_size("main")
	
	for i = 1, size do
		local stack = player_inv:get_stack("main", i)
		if not stack:is_empty() then
			local leftover = chest_inv:add_item("main", stack)
			player_inv:set_stack("main", i, leftover)
			
			if leftover:get_count() < stack:get_count() then
				moved = moved + 1
			end
		end
	end
	
	return moved > 0, moved
end

local function quick_move_from_chest(player, chest_pos)
	local player_inv = player:get_inventory()
	local chest_meta = minetest.get_meta(chest_pos)
	local chest_inv = chest_meta:get_inventory()
	
	if not chest_inv then return false, 0 end
	
	local moved = 0
	local size = chest_inv:get_size("main")
	
	for i = 1, size do
		local stack = chest_inv:get_stack("main", i)
		if not stack:is_empty() then
			local leftover = player_inv:add_item("main", stack)
			chest_inv:set_stack("main", i, leftover)
			
			if leftover:get_count() < stack:get_count() then
				moved = moved + 1
			end
		end
	end
	
	return moved > 0, moved
end

minetest.register_chatcommand("sort", {
	description = "Sort your inventory",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		if sort_inventory(player, "main") then
			return true, "Inventory sorted!"
		else
			return false, "Failed to sort inventory."
		end
	end
})

minetest.register_chatcommand("stack", {
	description = "Stack all items in inventory",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		if stack_all(player, "main") then
			return true, "Items stacked!"
		else
			return false, "Failed to stack items."
		end
	end
})

minetest.register_chatcommand("sortstack", {
	description = "Sort and stack inventory",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		stack_all(player, "main")
		sort_inventory(player, "main")
		return true, "Inventory sorted and stacked!"
	end
})

local chest_nodes = {
	"default:chest",
	"default:chest_locked",
	"default:chest_open",
	"mcl_chests:chest",
	"mcl_chests:chest_small",
	"mcl_chests:chest_left",
	"mcl_chests:chest_right",
	"mcl_chests:ender_chest",
	"mcl_chests:ender_chest_small",
	"mcl_chests:trapped_chest",
	"mcl_chests:trapped_chest_small",
	"mcl_chests:trapped_chest_left",
	"mcl_chests:trapped_chest_right",
	"mcl_chests:chest_locked",
	"mcl_barrels:barrel_closed",
	"mcl_barrels:barrel_open",
	"protector:chest",
}

local function find_nearest_chest(player)
	local eye_pos = player:get_pos()
	eye_pos.y = eye_pos.y + (player:get_properties().eye_height or 1.625)
	local look_dir = player:get_look_dir()
	local target_pos = vector.add(eye_pos, vector.multiply(look_dir, 5))
	
	local search_radius = {x=3, y=3, z=3}
	local nodes_in_range = minetest.find_nodes_in_area(
		vector.subtract(target_pos, search_radius),
		vector.add(target_pos, search_radius),
		chest_nodes
	)
	
	if #nodes_in_range == 0 then
		return nil
	end
	
	local player_pos = player:get_pos()
	local closest = nodes_in_range[1]
	local min_dist = vector.distance(player_pos, closest)
	
	for i = 2, #nodes_in_range do
		local dist = vector.distance(player_pos, nodes_in_range[i])
		if dist < min_dist then
			min_dist = dist
			closest = nodes_in_range[i]
		end
	end
	
	return closest
end

local function show_inventory_menu(player)
	local name = player:get_player_name()
	
	local formspec = {
		"formspec_version[4]",
		"size[10,8]",
		"bgcolor[#1E1E1ECC;true]",
		"label[0.5,0.5;Inventory Manager]",
		"button[0.5,1.5;2.8,0.8;sort;Sort]",
		"button[3.5,1.5;2.8,0.8;stack;Stack]",
		"button[6.5,1.5;2.8,0.8;sortstack;Sort & Stack]",
		"label[0.5,2.8;Chest Actions:]",
		"button[0.5,3.3;4.3,0.8;deposit;→ Deposit All]",
		"button[5.2,3.3;4.3,0.8;withdraw;← Withdraw All]",
		"label[0.5,4.5;Info:]",
		"label[0.5,5.0;• Sort: Organizes by category]",
		"label[0.5,5.4;• Stack: Combines duplicates]",
		"label[0.5,5.8;• Look at a chest for deposit/withdraw]",
		"button_exit[3.5,7;3,0.8;close;Close]"
	}
	
	minetest.show_formspec(name, "manage_inventory:menu", table.concat(formspec))
end

minetest.register_chatcommand("inv", {
	description = "Open Inventory Manager",
	func = function(name)
		local player = minetest.get_player_by_name(name)
		if not player then return false, "Player not found." end
		
		show_inventory_menu(player)
		return true
	end
})

minetest.register_on_player_receive_fields(function(player, formname, fields)
	if formname ~= "manage_inventory:menu" then return end
	
	local name = player:get_player_name()
	
	if fields.sort then
		sort_inventory(player, "main")
		minetest.chat_send_player(name, minetest.colorize("#00FF00", "✓ Inventory sorted!"))
		show_inventory_menu(player)
		
	elseif fields.stack then
		stack_all(player, "main")
		minetest.chat_send_player(name, minetest.colorize("#00FF00", "✓ Items stacked!"))
		show_inventory_menu(player)
		
	elseif fields.sortstack then
		stack_all(player, "main")
		sort_inventory(player, "main")
		minetest.chat_send_player(name, minetest.colorize("#00FF00", "✓ Sorted & stacked!"))
		show_inventory_menu(player)
		
	elseif fields.deposit or fields.withdraw then
		local chest_pos = find_nearest_chest(player)
		
		if not chest_pos then
			minetest.chat_send_player(name, minetest.colorize("#FF5555", "✗ No chest found! Look at a chest."))
			show_inventory_menu(player)
			return
		end
		
		if fields.deposit then
			local success, count = quick_move_to_chest(player, chest_pos)
			if success then
				minetest.chat_send_player(name, 
					minetest.colorize("#00FF00", string.format("✓ Deposited %d stack(s)!", count)))
			else
				minetest.chat_send_player(name, 
					minetest.colorize("#FF5555", "✗ Chest is full!"))
			end
		else
			local success, count = quick_move_from_chest(player, chest_pos)
			if success then
				minetest.chat_send_player(name, 
					minetest.colorize("#00FF00", string.format("✓ Withdrew %d stack(s)!", count)))
			else
				minetest.chat_send_player(name, 
					minetest.colorize("#FF5555", "✗ No items in chest!"))
			end
		end
		
		show_inventory_menu(player)
	end
end)

minetest.register_on_leaveplayer(function(player)
	local name = player:get_player_name()
	favorites_cache[name] = nil
end)

minetest.register_on_joinplayer(function(player)
	minetest.after(2, function()
		if not player or not player:is_player() then return end
		minetest.chat_send_player(player:get_player_name(), 
			minetest.colorize("#00FF00", "📦 [Manage Inventory] Use /inv for inventory tools!"))
	end)
end)

minetest.log("action", "[Manage Inventory] Loaded")