minetest.register_privilege("adminfill", {
    description = "Allows using heart fill, regeneration and extra heart commands",
    give_to_singleplayer = false
})

local active_regen = {}

local function get_max_hp(player)
    local extra = player:get_meta():get_int("extra_hearts")
    return 20 + extra
end

minetest.register_chatcommand("hf", {
    params = "[player]",
    description = "Fill hearts to max HP",
    privs = {adminfill = true},
    func = function(name, param)
        local target = param ~= "" and param or name
        local player = minetest.get_player_by_name(target)
        if not player then
            return false, "Player not found."
        end
        player:set_hp(get_max_hp(player))
        return true, "Hearts filled for " .. target
    end
})

minetest.register_chatcommand("he", {
    params = "[player] <hearts> <time>",
    description = "Give extra hearts for a limited time (seconds)",
    privs = {adminfill = true},
    func = function(name, param)
        local target, hearts, time = param:match("^(%S+)%s+(%d+)%s+(%d+)$")
        if not target then
            hearts, time = param:match("^(%d+)%s+(%d+)$")
            target = name
        end
        
        hearts, time = tonumber(hearts), tonumber(time)
        if not hearts or not time or hearts <= 0 or time <= 0 then
            return false, "Usage: /he [player] <hearts> <time>"
        end
        
        local player = minetest.get_player_by_name(target)
        if not player then
            return false, "Player not found."
        end
        
        local meta = player:get_meta()
        meta:set_int("extra_hearts", hearts * 2)
        meta:set_int("extra_hearts_time", time)
        
        local current_hp = player:get_hp()
        local new_max = get_max_hp(player)
        player:set_properties({hp_max = new_max})
        player:set_hp(new_max)
        
        return true, string.format("Granted %d extra hearts to %s for %d seconds", hearts, target, time)
    end
})

minetest.register_chatcommand("hr", {
    params = "[player] <amount> <time>",
    description = "Regenerate hearts over time (HP per second, 2 HP = 1 heart)",
    privs = {adminfill = true},
    func = function(name, param)
        local target, amount, time = param:match("^(%S+)%s+(%d+)%s+(%d+)$")
        if not target then
            amount, time = param:match("^(%d+)%s+(%d+)$")
            target = name
        end
        
        amount, time = tonumber(amount), tonumber(time)
        if not amount or not time or amount <= 0 or time <= 0 then
            return false, "Usage: /hr [player] <amount> <time>"
        end
        
        local player = minetest.get_player_by_name(target)
        if not player then
            return false, "Player not found."
        end
        
        local meta = player:get_meta()
        meta:set_int("regen_amount", amount * 2)
        meta:set_int("regen_time", time)
        meta:set_int("regen_remaining", time)
        active_regen[target] = true
        
        return true, string.format("Regenerating %d hearts per second for %s for %d seconds", amount, target, time)
    end
})

local timer = 0
minetest.register_globalstep(function(dtime)
    timer = timer + dtime
    if timer < 1 then return end
    timer = 0
    
    for _, player in ipairs(minetest.get_connected_players()) do
        local name = player:get_player_name()
        local meta = player:get_meta()
        local remaining = meta:get_int("regen_remaining")
        
        if remaining > 0 then
            local hp = player:get_hp()
            local max_hp = get_max_hp(player)
            if hp < max_hp then
                local amount = meta:get_int("regen_amount")
                player:set_hp(math.min(hp + amount, max_hp))
            end
            meta:set_int("regen_remaining", remaining - 1)
            
            if remaining - 1 <= 0 then
                active_regen[name] = nil
            end
        end
    end
end)

local extra_timer = 0
minetest.register_globalstep(function(dtime)
    extra_timer = extra_timer + dtime
    if extra_timer < 1 then return end
    extra_timer = 0
    
    for _, player in ipairs(minetest.get_connected_players()) do
        local meta = player:get_meta()
        local t = meta:get_int("extra_hearts_time")
        
        if t > 0 then
            meta:set_int("extra_hearts_time", t - 1)
            
            if t - 1 <= 0 then
                meta:set_int("extra_hearts", 0)
                player:set_properties({hp_max = 20})
                player:set_hp(math.min(player:get_hp(), 20))
                minetest.chat_send_player(player:get_player_name(), "Extra hearts expired!")
            end
        end
    end
end)

minetest.register_on_joinplayer(function(player)
    local meta = player:get_meta()
    local extra = meta:get_int("extra_hearts")
    if extra > 0 then
        player:set_properties({hp_max = 20 + extra})
    else
        player:set_properties({hp_max = 20})
    end
end)