-- API FILE

-- Does not include:
    -- Sonic screwdriver (sonic.lua)
    -- Exterior registering (tardis/exterior_api.lua) 

-- Does include:
    -- Get exterior position
    -- 4dir tools
    -- logging function
    -- Verify mods for interior schem
    -- Initiate interior doors
    -- Place Tardis interior
    -- Rebuild Tardis interior
    -- Link in_door to Tardis (custom interior pos)
    -- Locate Tardis
    -- Is this node a Time Rotor?
    -- Switch a Time Rotor on/off
    -- Update global table of all Tardis positions
    -- Validate interior position
    -- Lock/Unlock Tardis exit door
    -- Toggle antigrav system

--Planned:
    -- Summon Tardis API
    -- Waypoints
    -- Power
    -- and more

local data = core.get_mod_storage()
local S, NS = core.get_translator("drwho_tardis")
drwho_tardis.get = {}

-- Returns the default depth Tardis interiors are placed at
function drwho_tardis.get:interior_depth()
    return _drwho_tardis.interiorDepth
end

-- Returns the interior type of a player's Tardis 
function drwho_tardis.get:interior_type(id)
    local user = _drwho_tardis.get_user(id)
    local type = user.console_room.type or "old"
    return type
end

-- Returns the exterior position of a player's Tardis (locate_tardis)
function drwho_tardis.get:tardis_ext_position(id)
    local user = _drwho_tardis.get_user(id)
    if user.out_pos == nil or user.out_pos == "" then
        return false, "They don't have a Tardis!", nil -- sucess, errmsg, value
    end
    return true, "", user.out_pos
end 

-- Returns the Tardis exterior list, or an empty table if for some reason "exterior_list" doesn't exist
function drwho_tardis.get:exterior_list()
    local ext_list = core.deserialize(data:get_string("exterior_list")) or {}
    return ext_list
end

-- Returns exterior information based on it's index in the exterior_list
function drwho_tardis.get:exterior_info(index)
    local ext_list = drwho_tardis.get:exterior_list()
    return ext_list[index] -- if it's nil that's sad
end

-- Returns the current power levels of a player's Tardis
function drwho_tardis.get:power(id)
    local user = _drwho_tardis.get_user(id)
    return true, user.power, nil
end

---- Useful Functions ----

-- Takes a 4dir param2 value (0 - 3) and returns it as a 360* degree
function drwho_tardis.fourDir_to_dir(dir)
    local degree = dir * 90
    return degree
end

-- Takes a 4dir param2 value (0 - 3) and returns it as a 360* degree, flipped by 180 if param2 is even.
-- e.g. 0 (facing N) becomes 0 then 180. 1 (facing E) becomes 90 but not flipped by 180.
-- I'm unsure why I need to flip even 4dirs but not the others, but this is the only way I got it to work.
function drwho_tardis.fourDir_to_dir_flipped(dir)
    local degree = drwho_tardis.fourDir_to_dir(dir)
    if dir == 0 or dir == 2 then degree = degree + 180 end
    if degree >= 360 then degree = degree - 360 end
    return degree
end

-- Takes a position and a 4dir param2 value (0 - 3). 
-- Returns that position shifted by 1 node in front of it 
function drwho_tardis.shift_1_node_from_4dir(pos, dir)
    if dir == 0 then -- N
        pos.z = pos.z - 1
    elseif dir == 1 then -- E
        pos.x = pos.x - 1
    elseif dir == 2 then -- S
        pos.z = pos.z + 1
    elseif dir == 3 then -- W
        pos.x = pos.x + 1
    end
    return pos
end

-- Accepts either a table or 3 coordinate numbers.
function drwho_tardis.serialize_pos(pos, posy, posz)
    if type(pos) == "table" then
        return '('..pos.x..', '..pos.y..', '..pos.z..')'
    else -- Passed 3 numbers for x y z
        return '('..pos..', '..posy..', '..posz..')'
    end
end

-------------------------------------------------------------------------------------------------------
----------------------      PRIVATE API FUNCTIONS       -----------------------------------------------
-------------------------------------------------------------------------------------------------------

-- Makes debug logging much easier, and automatically disabled on release versions
-- While developing the mod, ensure that VERSION is appended with '-dev
function _drwho_tardis.log(input)
    if string.find(drwho_tardis.VERSION, 'dev') then -- print debug logs
        core.log('[drwho_tardis] '..input)
    end
end

-- Verifies that all required mods for a Tardis interior schematic exist
function _drwho_tardis:verify_mods_for_interior(type)
    local required_mods = _drwho_tardis.rooms[type].requires
    local verified = true
    local missing_mods = {}
    for _, mod in ipairs(required_mods) do 
        local exists = core.get_modpath(mod)
        if exists == nil then 
            verified = false
            missing_mods[#missing_mods+1] = mod
        end
    end
    return verified, "", missing_mods
end

-- [PRIVATE] Places a Tardis interior for a player (console room)
function _drwho_tardis:place_tardis_interior(id, type, pos)
    if pos == nil then 
        return false, "Position not provided!", nil
    else
        local user = _drwho_tardis.get_user(id) -- get user data
        if type == "custom" then
            -- Assume that the given pos is where the door is going to be
            _drwho_tardis:init_interior_door(id, pos) -- Start door timer, save metadata
            user.console_room.type = type -- Save the fact their interior is custom
            user.console_room.exit_door_pos = pos -- Remember the door position
            user.console_room.exit_door_offset = vector.new(0,0,0) -- there is no offset
            user.console_room.anchor_pos = pos -- 0 offset from the door pos, is the door pos
            _drwho_tardis.update_tardis_global_table(id, pos) -- Save anchor pos in global table
            user.in_pos = pos -- save new interior position
            user.r_pos = "" -- clear r_pos
            _drwho_tardis.save_user(id, user) -- save user data
            return true, "", pos -- success, message, door position
        end
        
        -- Ensure interior does not overlap with someone elses'
        pos.y = _drwho_tardis.interiorDepth
        local is_pos_valid = _drwho_tardis.validate_interior_pos(id, pos)
        if not is_pos_valid then
            _drwho_tardis.log('is_pos_valid is false')
            while is_pos_valid == false do
                pos.x = pos.x + 50
                _drwho_tardis.log('added 50 to pos.x')
                is_pos_valid = _drwho_tardis.validate_interior_pos(id, pos)
            end
        end

        local exit_door_offset = vector.new(0,0,0)
        
        if _drwho_tardis.rooms[type] then
            local are_mods_verified, _, missing_mods = _drwho_tardis:verify_mods_for_interior(type)
            if are_mods_verified == false then
                local msg = "This interior schematic requires additional mods: "
                for _, mod in ipairs(missing_mods) do msg = msg..mod.."  " end
                return false, msg, type
            end
        end 

        if _drwho_tardis.rooms[type] then
			core.place_schematic(pos, _drwho_tardis.rooms[type].path) --core.get_modpath("drwho_tardis") .. "/schems/console_room.mts"
            exit_door_offset = _drwho_tardis.rooms[type].exit_door_offset --  = {x=7,y=2,z=16}
            local exit_door_pos = vector.add(pos, exit_door_offset) -- Schematic adjustments so that the player gets teleported right in front of the door  
            user.console_room.type = type -- Save what type of interior they have
            user.console_room.exit_door_pos = exit_door_pos -- Remember the door position
            user.console_room.exit_door_offset = exit_door_offset -- Remember the door offset
            user.console_room.anchor_pos = pos -- Remember the schematic place anchor pos
            _drwho_tardis.update_tardis_global_table(id, pos) -- Save anchor pos in global table
            user.in_pos = exit_door_pos -- save new interior position
            _drwho_tardis.save_user(id, user) -- save user data
            return true, "", exit_door_pos -- success, message, door position 
        else
			return false, "Invalid interior room type! Use /tardis_int to change it.", type -- success, errmsg, int_type
        end
    end
end

-- id is the username of the player who owns the Tardis
function _drwho_tardis:init_interior_door(id, exit_door_pos)

    local d_meta = core.get_meta(exit_door_pos)
    local d_timer = core.get_node_timer(exit_door_pos)
    d_timer:start(0.2) --start door timer (in case it doesn't start on construct)
    d_meta:set_string("id", id) --set door id
    d_meta:set_string("type", "exit") -- set door type
    return true, exit_door_pos
end

-- [PRIVATE] Rebuilds a player's Tardis (rebuild_tardis)
-- Can fix old interiors being too high up in the world, and fix caves running through them
function _drwho_tardis:rebuild_tardis_interior(id)
    local user = _drwho_tardis.get_user(id) -- get user data

    local in_pos = table.copy(user.in_pos)
    if in_pos == "" or in_pos == nil then 
        return false, "You don't have a Tardis!" -- success, errmsg
    end

    local type = user.console_room.type
    if type=="" or type==nil then type = "default" end -- defaults
    local exit_door_offset = table.copy(user.console_room.exit_door_offset)
    if exit_door_offset == nil or exit_door_offset == "" or exit_door_offset == {} then
        exit_door_offset = _drwho_tardis.rooms[type].exit_door_offset
        user.console_room.exit_door_offset = exit_door_offset
    end
    local exit_door_offset = user.console_room.exit_door_offset
    if exit_door_offset=="" or exit_door_offset==nil then exit_door_offset = {x=7,y=2,z=16} end -- defaults

    -- This ensures that the old and new line up properly
    local anchor_pos = vector.subtract(in_pos, exit_door_offset)
    local success, msg, exit_door_pos = _drwho_tardis:place_tardis_interior(id, type, anchor_pos)

    if success == false then 
        _drwho_tardis.save_user(id, user) -- save user data
        return false, msg, exit_door_pos -- exit_door_pos is actually the type in this situation after returned by the place API function
    end

    -- Interior door stuff
    _drwho_tardis:init_interior_door(id, exit_door_pos)

    -- Clear Time Rotor pos because that doesn't exist anymore
    _drwho_tardis.update_one_value(id, "r_pos", "")

    return true, "", exit_door_pos -- success!
end

-- Check if the Time Rotor is there or not
-- name is required, pos is not
function _drwho_tardis.is_time_rotor(id, pos)
    local user = _drwho_tardis.get_user(id) -- get user data
    local r_pos
    if pos then 
        r_pos = pos
    else
        if not user.r_pos or user.r_pos == "" or user.r_pos == {} then 
            return false, "r_pos doesn't exist", user.r_pos
        else
            r_pos = table.copy(user.r_pos)
        end
    end
    
    local rmeta = core.get_meta(r_pos)
    local rotor_node = core.get_node(r_pos)
    
    if core.get_item_group(rotor_node.name, 'tardis_time_rotor') == 1 or 
        core.get_item_group(rotor_node.name, 'tardis_time_rotor_active') == 1 then
        return true, "rotor is there", r_pos
    else
        return false, "rotor is not there", r_pos
    end
end

-- Saves tardis interior position in global reference table
-- anchor_pos is the bottom-leftmost node where the schematic is placed from. 
function _drwho_tardis.update_tardis_global_table(id, anchor_pos)
    local pos_table = core.deserialize(data:get_string("_tardis_pos_table")) or {}
    if anchor_pos == "" or anchor_pos == nil or anchor_pos.x == nil or anchor_pos.y == nil or anchor_pos.z == nil then 
        return false 
    else
        pos_table[id] = anchor_pos
        data:set_string("_tardis_pos_table", core.serialize(pos_table))
        _drwho_tardis.log('tardis global table has been updated with '..id..'\'s Tardis anchor_pos')
    end
end

-- Validates if the given position is inside an existing Tardis interior
-- Iterates through the global tardis table 
-- most interiors are 42^3, we check +45 and -45 from every interior's schem anchor_pos so no nodes are able to overlap.
-- returns:
--      true if pos is not close to existing Tardis interiors
--      false if pos is too close and would overlap/partially destroy someone elses'
function _drwho_tardis.validate_interior_pos(id, pos)
    local pos_table = core.deserialize(data:get_string("_tardis_pos_table")) or {}
    local is_valid = true
    for t_id, t_pos in pairs(pos_table) do
        if t_id == id then _drwho_tardis.log('not checking against '..id..'\'s own tardis (its old pos)') goto continue end -- skip this user's own tardis
        if not t_pos then _drwho_tardis.log('this t_pos is nil!') goto continue end -- dont be nil
        _drwho_tardis.log('checking '..id..'\'s provided pos against '..t_id..'\'s tardis pos')
        if       pos.x >= t_pos.x-45 and pos.x <= t_pos.x+45
            and  pos.y >= t_pos.y-45 and pos.y <= t_pos.y+45
            and  pos.z >= t_pos.z-45 and pos.z <= t_pos.z+45 
        then
            is_valid = false
            break
        end
        ::continue::
    end
    _drwho_tardis.log('[drwho_tardis] validate interior pos is '..tostring(is_valid))
    return is_valid
end

-- Starts/stops time rotor animation
-- send_msg parameter is optional, defaults to true. 
function _drwho_tardis.switch_time_rotor(id, on_off, send_msg)
    local user = _drwho_tardis.get_user(id) -- get user data
	-- 'r' is the Rotor
    local is_it_there, msg = _drwho_tardis.is_time_rotor(id)

    if is_it_there then
        local r_pos = table.copy(user.r_pos)
        local rmeta = core.get_meta(r_pos)
        local style = rmeta:get_string("style")
        local rotor_node = core.get_node(r_pos)
        if on_off == "on" then 
            core.swap_node(r_pos, {name = "drwho_tardis:rotor_active"..style })
            return true, r_pos
        elseif on_off == "off" then
            core.swap_node(r_pos, {name = "drwho_tardis:rotor"..style })
            return true, r_pos
        end
    else
        if send_msg == nil or send_msg ~= false then
            core.chat_send_player(id, S("You need to have a Time Rotor!")) 
        end
        return false, r_pos
    end
end

-- locks or unlocks the interior door. if on_off is not given, it will toggle.
function _drwho_tardis.toggle_int_door(id, on_off)
    local user = _drwho_tardis.get_user(id) -- get user data
    local in_pos = table.copy(user.in_pos)

    if on_off == "lock" then
        core.set_node(in_pos, { name = "drwho_tardis:in_door_locked" }) 
        local dmeta = core.get_meta(in_pos)
        dmeta:set_string("id", id) -- set door id
        dmeta:set_string("type", "exit") -- set door type 
        return true, "Interior door locked"
    elseif on_off == "unlock" then 
        core.set_node(in_pos, { name = "drwho_tardis:in_door" })
        -- Start door timer
        local dtimer = core.get_node_timer(in_pos)
        local dmeta = core.get_meta(in_pos)
        dmeta:set_string("id", id) -- set door id 
        dmeta:set_string("type", "exit") -- set door type 
        dtimer:start(0.2) -- start door timer (in case it doesn't start on construct)
        return true, "Interior door unlocked"
    else -- on_off is not given, toggle door
        if core.get_node(in_pos).name == "drwho_tardis:in_door" then
            core.set_node(in_pos, { name = "drwho_tardis:in_door_locked" }) 
            local dmeta = core.get_meta(in_pos)
            dmeta:set_string("id", id) -- set door id
            dmeta:set_string("type", "exit") -- set door type 
           return true, "Interior door locked"
        else
            core.set_node(in_pos, { name = "drwho_tardis:in_door" })
            -- Start door timer
            local dtimer = core.get_node_timer(in_pos)
            local dmeta = core.get_meta(in_pos)
            dmeta:set_string("id", id) -- set door id 
            dmeta:set_string("type", "exit") -- set door type 
            dtimer:start(0.2) -- start door timer (in case it doesn't start on construct)
            return true, "Interior door unlocked"
        end
    end

end

-- Toggle the antigrav system
-- returns success, status
function _drwho_tardis.toggle_mavity(id)
    local user = _drwho_tardis.get_user(id) -- get user data
    local msg
    if user.respect_mavity == "yes" then -- It's on, turn it off
        user.respect_mavity = "no"
        msg = "Antigrav system disabled"
    elseif user.respect_mavity == "no" then -- It's off, set to sink
        user.respect_mavity = "sink" -- sink in liquids mode
        msg = "Antigrav system enabled and set to sink in liquids"
    else -- it's "sink" or nil; turn it on
        user.respect_mavity = "yes"
        msg = "Antigrav system enabled but won't sink in water"
    end
    _drwho_tardis.save_user(id, user) -- save user data
    return true, msg
end