-- CONTENTS
--    Detect GAMETYPE
--    Interior rooms def 
--	  Check if data needs to be migrated
-- 	  Initiate userdata function
--	  Migrate userdata function
--    Crafting aliases for different games 
--    Store itemstrings of our items

local data = core.get_mod_storage()
local S, NS = core.get_translator("drwho_tardis")

-- DETECT GAME TYPE (MTG, MCL, RP)
_drwho_tardis.GAMETYPE = "???"
if core.get_modpath("default") then 
	_drwho_tardis.GAMETYPE = "mtg"
elseif core.get_modpath("mcl_core") then 
	_drwho_tardis.GAMETYPE = "mcl"
elseif core.get_modpath("rp_default") then 
	_drwho_tardis.GAMETYPE = "rp"
else 
	_drwho_tardis.GAMETYPE = "other"
end

-- Reset exterior list 
-- if this is not done, every time the server is restarted it will add them again
-- eventually overcrowding the list and looking bad
data:set_string("ext_list", core.serialize({}))

-- Interior depth
_drwho_tardis.interiorDepth = -300 -- default value
if _drwho_tardis.GAMETYPE == "mtg" then 
	_drwho_tardis.interiorDepth = -10000
elseif _drwho_tardis.GAMETYPE == "mcl" then
	_drwho_tardis.interiorDepth = -50 -- just above bedrock if placed at sea level
elseif _drwho_tardis.GAMETYPE == "rp" then
	_drwho_tardis.interiorDepth = -10000
end

-- pre-built interior console rooms. 
-- all new interiors MUST be 42^3 in size.
-- exit_door_offset is from where the schem is placed to where the exit door is
-- requires is a list of mods whose nodes are used in the interior, and may be used to ensure there are no unknown nodes. 
-- Although default interiors should strive to only use nodes from this mod.
_drwho_tardis.rooms = {
	old = { -- old original console room
		type = "old",
		path = core.get_modpath("drwho_tardis").."/schems/tardis_console_room_old.mts",
		exit_door_offset = {x=7,y=2,z=16},
		requires = {"drwho_tardis"},
	},
	default_22 = { -- old default in 1.2.0
		type = "default_22",
		-- size is 22x22x22
		path = core.get_modpath("drwho_tardis").."/schems/tardis_console_room_default.mts",
		exit_door_offset = {x=11,y=5,z=19},
		requires = {"drwho_tardis"},
	},
	default = { -- new default large interior with some pre-built rooms
		type = "default",
		-- size is 42x42x42
		path = core.get_modpath("drwho_tardis").."/schems/tardis_interior_large.mts",
		exit_door_offset = {x=12,y=5,z=19},
		requires = {"drwho_tardis"},
	},
	basic = { -- new large interior which only includes the console room
		type = "basic",
		-- size is 42x42x42
		path = core.get_modpath("drwho_tardis").."/schems/tardis_interior_large_basic.mts",
		exit_door_offset = {x=25,y=12,z=32},
		requires = {"drwho_tardis"},
	},
	green = { -- small console room using the green and grey metal from scifi_nodes
		type = "green",
		-- size is 42x42x42
		path = core.get_modpath("drwho_tardis").."/schems/tardis_scifi_green_large.mts",
		exit_door_offset = {x=16,y=13,z=24},
		requires = {"drwho_tardis", "scifi_nodes"},
	},
}


--------------------------
------- USER DATA --------
--------------------------
-- Adapted from https://github.com/Beanzilla/Tardis under the MIT license

-- Check if we need to migrate their data
function _drwho_tardis.do_we_need_to_migrate_data(user_name)
	-- we know if they used it previously because they have in_pos defined using old data struct
	local in_pos = _drwho_tardis.data:get_string(user_name.."in_pos")
	local data_migrated = _drwho_tardis.data:get_string(user_name.."data_migrated")
	_drwho_tardis.log("Checking if user data should be migrated from 1.2.0 -> 1.3.0+")
	if in_pos ~= "" and in_pos and data_migrated ~= "yes" then
		_drwho_tardis.log("Yes it does")
		return true
	else 
		_drwho_tardis.log("No it doesn't")
		return false
	end 
end

-- Obtains either an empty table or the user's data
function _drwho_tardis.get_user(user_name)
    local user = data:get_string(user_name)

	if user == nil or user == "" or user == {} then
        _drwho_tardis.log("Didn't find a user by name '"..user_name.."', creation occured")
		_drwho_tardis.log(user)
		user = _drwho_tardis.init_new_user(user_name, {})

		if _drwho_tardis.do_we_need_to_migrate_data(user_name) then
			user = _drwho_tardis.migrate_user_data(user_name, user)
			_drwho_tardis.log("Migrated user data with name '"..user_name.."' from data from pre-v1.3.0") 
		end
    else
        --_drwho_tardis.log("Found '"..user_name.."' with '"..user.."' data")
		_drwho_tardis.log("Retrieved user data")
        user = core.deserialize(user)
		--_drwho_tardis.log(core.serialize(user))
    end
	--_drwho_tardis.log("Returning '"..core.serialize(user).."'")
    return user
end

-- Saves the user's data
function _drwho_tardis.save_user(user_name, user_data)

    data:set_string(user_name, core.serialize(user_data))
	_drwho_tardis.log("Saved user data")
end

-- for convenience' sake
-- if only one or two values need to be updated, it's safer to use this rather than save the whole thing to avoid overwriting recent changes
-- key must be a string, but the same as it would usually. For user.r_pos, use 'r_pos'.
function _drwho_tardis.update_one_value(user_name, key, value)
    local user = _drwho_tardis.get_user(user_name)
	user[key] = value
	--_drwho_tardis.log(S("Set key '@1' to value '@2' in @3's user data.", key, value, user_name))
	_drwho_tardis.save_user(user_name, user)
end

-- Makes new user data
-- This can also be used as a reference to what the data structure is like
-- Must be given the user data when run otherwise Stack Overflow
function _drwho_tardis.init_new_user(user_name, user)
    if user.version ~= nil then
        _drwho_tardis.log("User '"..user_name.."' was found with data, rewriting data")
    end

    user.version = drwho_tardis.VERSION -- Know when a user's data was initiated for compatability's sake when breaking changes are made
    user.data_migrated = "no" -- Did they migrate from v1.2.1 to v1.3.0 ? "yes" or "no"; set by _drwho_tardis.migrate_user_data(user_name, user)
	user.power = 3 -- max is 10. Used to power all Tardis functions, may be reworked sometime.
    user.has_a_tardis = false -- Set to True when they place their Tardis
	user.been_inside_tardis = false
	user.times_travelled_in_tardis = 0 -- Keep a tally for statistics and achievements
    user.waypoints = { -- Name and pos of waypoints. In future, will support custom names and more than 3.
		way1 = vector.new(0, 0, 0),
		way2 = vector.new(0, 0, 0),
		way3 = vector.new(0, 0, 0),
	}

	-- Navigation related values
    user.in_pos = vector.new(0, 0, 0) -- Where to spawn the user so they are inside the tardis (Default's to empty)
    user.out_pos = vector.new(0, 0, 0) -- Where to spawn the user when they exit the tardis (Default's to empty)
    user.prev_out_pos = vector.new(0, 0, 0) -- Stores last position (used for when dematerialized but then ran out of power)
    user.dest_pos = vector.new(0, 0, 0) -- Where the tardis is pointing to go to
	user.r_pos = "" -- Where the Time Rotor is
	user.travel_factor = 1 -- How much the levers increase the position by each time (1, 10, 100, 1000)
    user.respect_mavity = "yes" -- "yes", "no", "float". Controls how the Tardis adjusts it's position before landing.
    user.finding_biome = "" -- The most recent biome they travelled to, empty if they didn't use Find Biome

	user.exterior_skin = "drwho_tardis:tardis_blue" -- The outside of the Tardis. Previously 'look'

	-- We need to know:
	--	the type, that identifies what schematic was used to place it
	-- 	the difference in position between where we placed the schematic and where the door is
	-- 	where we placed the schematic (anchor_pos)
	user.console_room = {
		type = core.settings:get('drwho_tardis.default_interior') or 'default',
		exit_door_pos = "",
		exit_door_offset = "",
		anchor_pos = vector.new(0, 0, 0) -- previously console_place_pos. The position given to the place interior schematic function.
	},


    -- Save it
    _drwho_tardis.save_user(user_name, user)
    return user -- Be able to start using data straight away without having to get it again
end


-- Migrate user data from v1.2.1 to v1.3.0
-- Assumes that init_new_user was run beforehand
-- Must be given the user data when run otherwise Stack Overflow
function _drwho_tardis.migrate_user_data(user_name, user)
	local id = user_name
	if user == "" or user == nil then user = {} end
    if user.version ~= nil then
        _drwho_tardis.log("User '"..user_name.."' was found with data, rewriting data")
    end

    user.version = drwho_tardis.VERSION
    user.power = data:get_int(id.."power")
    user.has_a_tardis = true
	user.been_inside_tardis = true
	user.times_travelled_in_tardis = 0
    user.waypoints = {
		way1 = core.deserialize(data:get_string(id.."way1")),
		way2 = core.deserialize(data:get_string(id.."way2")),
		way3 = core.deserialize(data:get_string(id.."way3")),
	}

	-- Navigation related values
    user.in_pos = core.deserialize(data:get_string(id.."in_pos"))
    user.out_pos = core.deserialize(data:get_string(id.."out_pos"))
    user.prev_out_pos = core.deserialize(data:get_string(id.."out_pos"))
    user.dest_pos = vector.new(data:get_int(id.."x_dest"), data:get_int(id.."y_dest"), data:get_int(id.."z_dest"))
	user.r_pos = core.deserialize(data:get_string(id.."r_pos"))
	user.travel_factor = data:get_string(id.."factor")
    user.respect_mavity = core.deserialize(data:get_string(id.."respect_mavity")) or "yes"
    user.finding_biome = "" 

	user.exterior_skin = data:get_string(id.."look")

	local type = data:get_string(id.."type")
	if type == "default" then type = "default_22" end

	user.console_room = {
		type = type, -- see above
		exit_door_pos = core.deserialize(data:get_string(id.."door_pos_table")).door_0,
		exit_door_offset = core.deserialize(data:get_string(id.."door_offset_table")).door_0,
		anchor_pos = core.deserialize(data:get_string(id.."console_place_pos")),
	},

	-- add to global table of all user's Tardis interiors, based on their schematic anchor_pos (place_pos)
	_drwho_tardis.update_tardis_global_table(id, core.deserialize(data:get_string(id.."console_place_pos")))
	

	-- Remember that their data has been migrated
	data:set_string(id.."data_migrated", "yes")
	user.data_migrated = "yes"

    -- Save it
    _drwho_tardis.save_user(user_name, user)
    return user -- Be able to start using data straight away without having to get it again
end



-- CRAFTING ALIASES
-- note that all crafting is going to be rethought and reworked at some point
-- but for now the equivelant items and nodes in different games are saved
-- in MCL, all console nodes use flowers as the special item

_drwho_tardis.items = {}
local i = _drwho_tardis.items -- make it easier to write and read

if _drwho_tardis.GAMETYPE == "mcl" then -- MINECLONE ITEMS
	i.ingot_steel 		= "mcl_core:iron_ingot"
	i.ingot_copper 		= "mcl_copper:copper_ingot"
	i.ingot_tin 		= "mcl_core:iron_ingot" -- They don't have tin, use iron instead
	i.lump_tin			= "mcl_core:iron_nugget"
	i.ingot_gold 		= "mcl_core:gold_ingot"
	i.diamond 			= "mcl_core:diamond"
	i.crystal 			= "mesecons:wire_00000000_off" -- Redstone, but has a weird itemstring
	i.coal				= "mcl_core:coal_lump"
	i.dirt				= "mcl_core:dirt"
	i.stone				= "mcl_core:stone"

	i.block_steel 		= "mcl_core:ironblock" -- steel is iron
	i.block_copper 		= "mcl_copper:block"
	i.block_tin 		= "mcl_core:ironblock" -- tin becomes iron
	i.block_bronze		= "mcl_copper:block" -- bronze becomes copper -- also O item
	i.block_gold 		= "mcl_core:goldblock"
	i.block_diamond 	= "mcl_core:diamondblock"
	i.block_crystal 	= "mesecons_torch:redstoneblock"

	i.stick 			= "mcl_core:stick"
	i.silver_sand 		= "mcl_core:sand"
	i.cactus 			= "mcl_flowers:tulip_orange" -- C item
	i.grass_dry 		= "mcl_flowers:tulip_white" -- X item
	i.grass_jungle 		= "mcl_flowers:allium" -- Z item
	i.grass 			= "mcl_flowers:poppy" -- Y item
	i.silver_sandstone 	= "mcl_flowers:azure_bluet" -- GO item
	i.glass 			= "mcl_core:glass"
	i.obsidian 			= "mcl_core:obsidian" -- F item
	i.flower			= "mcl_flower:dandelion" -- S item
	i.red 				= "mcl_dye:red" -- Sonic Lipstick

elseif _drwho_tardis.GAMETYPE == "rp" then -- REPIXTURE ITEMS
	i.ingot_steel 		= "rp_default:ingot_steel"
	i.ingot_copper 		= "rp_default:ingot_copper"
	i.ingot_tin 		= "rp_default:ingot_tin"
	i.lump_tin			= "rp_default:tin_lump"
	i.ingot_gold 		= "rp_gold:ingot_gold"
	i.diamond 			= "rp_jewels:jewel"
	i.crystal 			= "rp_lumien:crystal_off"
	i.coal				= "rp_default:lump_coal"
	i.dirt				= "rp_default:dirt"
	i.stone				= "rp_default:stone"

	i.block_steel 		= "rp_default:block_steel"
	i.block_copper 		= "rp_default:block_copper"
	i.block_tin 		= "rp_default:block_tin"
	i.block_bronze		= "rp_default:block_bronze"  -- O item
	i.block_gold 		= "rp_gold:block_gold"
	i.block_diamond 	= "rp_jewels:jewel"
	i.block_crystal 	= "rp_lumien:block"

	i.stick 			= "rp_default:stick"
	i.cactus 			= "rp_default:cactus" -- C item
	i.grass_dry 		= "rp_default:dry_grass" -- X item
	i.grass_jungle 		= "rp_default:fern" -- Z item
	i.grass 			= "rp_default:grass" -- Y item
	i.silver_sand 		= "rp_default:sand" 
	i.silver_sandstone 	= "rp_default:sandstone" -- GO item
	i.glass 			= "rp_default:glass"
	i.obsidian 			= "rp_default:ingot_bronze" -- F item
	i.flower			= "rp_default:flower" -- S item
	i.red 				= "rp_default:apple" -- Sonic Lipstick

else -- MINETEST GAME ITEMS, and default to MTG to avoid stupid bugs relating to them being undefined

	i.ingot_steel 		= "default:steel_ingot"
	i.ingot_copper 		= "default:copper_ingot"
	i.ingot_tin 		= "default:tin_ingot"
	i.lump_tin			= "default:tin_lump"
	i.ingot_gold 		= "default:gold_ingot"
	i.diamond 			= "default:diamond"
	i.crystal 			= "default:mese_crystal"
	i.coal				= "default:coal_lump"
	i.dirt				= "default:dirt"
	i.stone				= "default:stone"

	i.block_steel 		= "default:steelblock"
	i.block_copper 		= "default:copperblock"
	i.block_tin 		= "default:tinblock"
	i.block_bronze		= "default:bronzeblock" -- C item
	i.block_gold 		= "default:goldblock"
	i.block_diamond 	= "default:diamondblock"
	i.block_crystal 	= "default:mese"

	i.stick 			= "default:stick"
	i.cactus 			= "default:cactus" -- C item
	i.grass_dry 		= "default:dry_grass_1" -- X item 
	i.grass_jungle 		= "default:junglegrass" -- Z item
	i.grass 			= "default:grass_1" -- Y item
	i.silver_sand 		= "default:silver_sand"
	i.silver_sandstone 	= "default:silver_sandstone_block" -- GO item
	i.glass 			= "default:glass"
	i.obsidian 			= "default:obsidian" -- F item
	i.flower			= "flowers:dandelion_white" -- S item
	i.red 				= "dye:red" -- Sonic Lipstick
end


--
-- DrWho Items table
--

-- Access the full itemstrings of our items easier
_drwho_tardis.drwho_items = { 
    -- Our craftitems:
    arton_crystal = "drwho_tardis:arton_crystal",
    spacetime = "drwho_tardis:spacetime",
    circuitry_board = "drwho_tardis:circuitry_board",
    azbantium_shard = "drwho_tardis:azbantium_shard",
    dalekanium_ingot = "drwho_tardis:dalekanium_ingot",
    -- Nodes crafted:
    tardis = "drwho_tardis:tardis",
    lab = "drwho_tardis:lab",
    vortex_manipulator = "drwho_tardis:vortex_manipulator",
    rassilon_gauntlet = "drwho_tardis:rassilon_gauntlet",
    chest = "drwho_tardis:chest", -- Gallifreyan Chest
    rotor = "drwho_tardis:rotor",
    console_x = "drwho_tardis:console_x",
    console_y = "drwho_tardis:console_y",
    console_z = "drwho_tardis:console_z",
    console_go = "drwho_tardis:console_go",
    console_c = "drwho_tardis:console_c",
    console_f = "drwho_tardis:console_f",
    console_o = "drwho_tardis:console_o",
    console_s = "drwho_tardis:console_s",
    monitor = "drwho_tardis:monitor",
    light = "drwho_tardis:light",
    azbantium = "drwho_tardis:azbantium",
    azbantium_shard = "drwho_tardis:azbantium_shard",
	turquiose = "drwho_tardis:turquiose_block",
    grey_block = "drwho_tardis:grey_block",
    grey_block_slab = "drwho_tardis:grey_block_slab",
    walls = {
        one_circle = "drwho_tardis:wall_one_circle",
        one_circle_halfs = "drwho_tardis:wall_one_circle_halfs",
        wall = "drwho_tardis:wall_craftable",
        circles_16 = "drwho_tardis:wall_circles_16x16",
        circles_32 = "drwho_tardis:wall_circles_32x32",
    },
}